#if !defined( __MSDOS__ )
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <limits.h>
#include <string.h>
#include <errno.h>

#include <debug.h>

#if defined( USE_SLANG )
#include <slang.h>
#endif

#if defined( USE_CURSES )
#include <curses.h>
#endif

// messaggi
const char* MSG_HEAP = "Bad heap";
const char* MSG_PRECONDITION = "Precondition violated";
const char* MSG_ABSTRACT = "Abstract code";
const char* MSG_NOTIMP = "Code not implemented";
const char* MSG_RUNTIME = "Runtime error";
const char* MSG_CHECK = "Check failed";

// ****************************************************************************
// logging
// - attivo solo in ALPHA

#if __DEBUG >= __DEBUG_ALPHA

// massima dimensione di una linea di output
#define DEBUG_OUTPUT_MAX 128

static FILE* debug_log = 0;
static bool debug_log_need_close = false;

static void debug_log_init() {
  PRECONDITION( !debug_log );
  debug_log = fopen( "debug.log", "a" );
  if (debug_log) {
    time_t tm;
    time(&tm);
    fprintf( debug_log, "---------------------------------------------------------------------------\n");
    fprintf( debug_log, "-- compiled %s %s\n", __DATE__, __TIME__ );
    fprintf( debug_log, "-- started %s", ctime(&tm) );
    fflush( debug_log );
    debug_log_need_close = true;
  } else {
    debug_log = stderr;
    debug_log_need_close = false;
  }
}

static void debug_log_done() {
  fprintf( debug_log, "-- exit\n");
  fflush( debug_log );
  if (debug_log_need_close) {
    fclose( debug_log );
  }
}

#endif

// ****************************************************************************
// Trace
// - attivo solo in ALPHA

#if __DEBUG >= __DEBUG_ALPHA

void debug_trace(const char* file, unsigned line, const char* function) {
  fprintf(debug_log,"%s:%d:in %s:",file,(int)line,function);
  fprintf(debug_log,"\n");
  fflush(debug_log);
}

void debug_trace(const char* file, unsigned line, const char* function, const char* a) {
  fprintf(debug_log,"%s:%d:in %s:",file,(int)line,function);
  fprintf(debug_log,"<%s>\n",a);
  fflush(debug_log);
}

void debug_trace(const char* file, unsigned line, const char* function, int a) {
  fprintf(debug_log,"%s:%d:in %s:",file,(int)line,function);
  fprintf(debug_log,"<%d>\n",a);
  fflush(debug_log);
}

void debug_trace(const char* file, unsigned line, const char* function, const char* a, const char* b) {
  fprintf(debug_log,"%s:%d:in %s:",file,(int)line,function);
  fprintf(debug_log,"<%s,%s>\n",a,b);
  fflush(debug_log);
}

void debug_trace(const char* file, unsigned line, const char* function, const char* a, const char* b, const char* c) {
  fprintf(debug_log,"%s:%d:in %s:",file,(int)line,function);
  fprintf(debug_log,"<%s,%s,%s>\n",a,b,c);
  fflush(debug_log);
}

void debug_trace(const char* file, unsigned line, const char* function, const char* a, int b) {
  fprintf(debug_log,"%s:%d:in %s:",file,(int)line,function);
  fprintf(debug_log,"<%s,%d>\n",a,b);
  fflush(debug_log);
}

void debug_trace(const char* file, unsigned line, const char* function, const char* a, char b) {
  fprintf(debug_log,"%s:%d:in %s:",file,(int)line,function);
  fprintf(debug_log,"<%s,%c>\n",a,b);
  fflush(debug_log);
}

void debug_trace(const char* file, unsigned line, const char* function, const char* a, int b, const char* c) {
  fprintf(debug_log,"%s:%d:in %s:",file,(int)line,function);
  fprintf(debug_log,"<%s,%d,%s>\n",a,b,c);
  fflush(debug_log);
}

void debug_trace(const char* file, unsigned line, const char* function, const char* a, int b, int c) {
  fprintf(debug_log,"%s:%d:in %s:",file,(int)line,function);
  fprintf(debug_log,"<%s,%d,%d>\n",a,b,c);
  fflush(debug_log);
}

#endif

// ****************************************************************************
// Gestore avanzato dell'heap
// - attivo solo in ALPHA

#if __DEBUG >= __DEBUG_ALPHA

// -------------------------------------------------------------------------
// magic

// Elemento di allocazione
// Composto da 3 componenti
//   ) magic_header  marcatore di inizio blocco
//   ) data          dati effettivamente utilizzati
//   ) magic_footer  marcatore di fine blocco
struct magic {
    // magic_header header;
    // char data[];
    // magic_footer footer;
  };

// marcatore di inizio blocco
struct magic_header {
    unsigned size; // dimensione dei dati allocati
    unsigned line; // linea di sorgente
    unsigned count; // numero di allocazione
    const char* file; // file di sorgente, puntatore a costante stringa
    magic_header* pred; // lista doppia concatenata
    magic_header* next;
    unsigned crc; // crc delle informazioni contenute nell'header
    magic* tag; // puntatore al magic  
  };

// marcatore di fine blocco
struct magic_footer {
    magic* tag; // puntatore al magic
  };

// numero incrementale di allocazioni
unsigned magic_counter = 0;

// numero di allocazioni all'inizio della gestione dell'heap
static unsigned magic_counter_start;

// inizio della lista degli elementi allocati
static magic_header* magic_base = 0;

// converte magic in ptr (zona dati)
static inline const void* magic2ptr(const magic* m) {
  return ((const char*)m) + sizeof(magic_header);
}

static inline void* magic2ptr(magic* m) {
  return ((char*)m) + sizeof(magic_header);
}

// converte ptr in magic
static inline const magic* ptr2magic(const void* p) {
  return (const magic*)(((const char*)p) - sizeof(magic_header));
}

static inline magic* ptr2magic(void* p) {
  return (magic*)(((char*)p) - sizeof(magic_header));
}

// puntatore al campo header di una struttura magic
static inline const magic_header* magic2header(const magic* m) {
  return ((const magic_header*)m);
}

static inline magic_header* magic2header(magic* m) {
  return ((magic_header*)m);
}

static inline const magic* header2magic(const magic_header* h) {
  return ((const magic*)h);
}

static inline magic* header2magic(magic_header* h) {
  return ((magic*)h);
}

// puntatore al campo footer di una struttura magic
// in:
//   size dimensione in char dei dati allocati
static inline const magic_footer* magic2footer(const magic* m, unsigned size) {
  return (const magic_footer*)( ((const char*)m) + size + sizeof(magic_header) );
}

static inline magic_footer* magic2footer(magic* m, unsigned size) {
  return (magic_footer*)( ((char*)m) + size + sizeof(magic_header) );
}

// riempie completamente di spazzatura un magic
// in:
//   size dimensione in char dei dati allocati
static void magic_full_garbage_fill(magic* m, unsigned size) {
  size += sizeof(magic_header) + sizeof( magic_footer );
  void* p = m;
  static unsigned magic_seed = 0xAA;
  for(unsigned i=0;i<size;++i)
    ((unsigned char*)p)[i] = ++magic_seed;
}

// riempie completamente di spazzatura il campo data di un magic
// in:
//   size dimensione in char dei dati allocati
static void magic_data_garbage_fill(magic* m, unsigned size) {
  void* p = magic2ptr(m);
  static unsigned magic_seed = 0xAA;
  for(unsigned i=0;i<size;++i)
    ((unsigned char*)p)[i] = ++magic_seed;
}

// calcola il crc del'header
// in:
//   h header di cui calcolare il crc
static unsigned magic_crc_compute(const magic_header* h) {
  return h->size ^ ((unsigned)h->pred) ^ ((unsigned)h->next);
}

// imposta il campo di crc
// out:
//   m magic a cui impostare i crc
static void magic_crc_set(magic_header* h) {
  h->crc = magic_crc_compute( h );
}

// inserisce un magic nella lista
// in:
//   m magic da inserire nella lista
// out:
//   imposta il crc nell'header e footer di m e altri elementi nella lista
// note:
//   da chiamare SOLO quando tutti gli altri campi di header e footer sono 
//   stati impostati, con l'esclusione dei campi di crc
static void magic_insert(magic* m) {
  magic_header* h = magic2header( m );
  if (!magic_base) {
    // crea la lista
    magic_base = h;
    h->next = h;
    h->pred = h;
    // imposta i crc
    magic_crc_set( h );
  } else {
    // inserisce nella lista
    h->next = magic_base->next;
    h->pred = magic_base;
    magic_base->next->pred = h;
    magic_base->next = h;
    // imposta i crc
    magic_crc_set( h );
    magic_crc_set( h->pred );
    magic_crc_set( h->next );
  }
}

// elimina un magic dalla lista
// out:
//   imposta il crc nell'header e footer di elementi nella lista
static void magic_remove(magic* m) {
  magic_header* h = magic2header( m );
  if (h->next == h) {
    // cancella la lista
    CHECK( h->pred == h );
    CHECK( magic_base == h );
    magic_base = 0;
  } else {
    // eventualmente sposta l'inizio della lista
    if (magic_base == h)
      magic_base = magic_base->next;
    CHECK( magic_base != h );
    // elimina dalla lista
    h->pred->next = h->next;
    h->next->pred = h->pred;
    // imposta i crc
    magic_crc_set( h->pred );
    magic_crc_set( h->next );
  }
}

// test del magic
// return
//   true il magic  consistente
// note:
//   NULL NON  consistente
static bool magic_test(const magic* m) {
  if (!m) return false;
  const magic_header* h = magic2header(m);
  if (h->tag != m) return false;
  if (h->crc != magic_crc_compute(h)) return false;
  const magic_footer* f = magic2footer(m, h->size);
  if (f->tag != m) return false;
  return true;
}

// test dell' header
// return
//   true l'header e' corretto
// note:
//   NULL NON  consistente
static bool magic_header_test(const magic_header* h) {
  if (!h) return false;
  if (h->tag != header2magic(h)) return false;
  if (h->crc != magic_crc_compute(h)) return false;
  return true;
}

// test di tutti gli elementi allocati
static bool magic_test() {
  if (magic_base) {
    const magic_header* i = magic_base;
    do {
      if (!magic_test( header2magic(i))) 
        return false;
      i = i->next;
    } while (i!=magic_base);
  } 
  return true;
}

// test del puntatore
// return:
//   !=0 il puntatore  consistente
// note:
//   NULL NON  consistente
static bool magic_ptr_test(const void* p) {
  if (!p) return false;
  return magic_test( ptr2magic(p) );
}

// compone informazioni sul magic
static void magic_info_get(const magic* m, char* info, unsigned max) {
  PRECONDITION( max >= 128 );
  const magic_header* h = magic2header( m );
  if (!magic_header_test( h )) {
    sprintf( info, "ptr %p (invalid header)", magic2ptr(m) );
  } else if (!magic_test(m)) {
    sprintf( info, "%s:%d: ptr %p, %d byte, %d counter, (invalid footer)", h->file, h->line, magic2ptr(m), h->size, h->count );
  } else {
    sprintf( info, "%s:%d: ptr %p, %d byte, %d counter", h->file, h->line, magic2ptr(m), h->size, h->count );
  }
}

// compone informazioni sul puntatore
static void magic_ptr_info_get(const void* p, char* info, unsigned max) {
  if (!p) {
    sprintf( info, "unknow:0: null pointer" );
  } else
    magic_info_get( ptr2magic( p ), info, max );
}

// libera il puntatore
// note:
//   presuppone p validato da magic_ptr_test
static void magic_free(void* p) {
  magic* m = ptr2magic(p);
  magic_header* h = magic2header(m);
  // rimuve dalla lista
  magic_remove( m );
  // riempie di spazzatura TUTTO il magic
  magic_full_garbage_fill(m, h->size );
  // dealloca
  free( m );
}

// alloca il puntatore
// return:
//   !=0 puntatore allo spazio allocato
//   ==0 manca memoria
static void* magic_alloc(size_t size, const char* file, unsigned line) {
  magic* m = (magic*)malloc( size + sizeof(magic_header) + sizeof(magic_footer) );
  if (!m) return 0;
  magic_header* h = magic2header(m);
  magic_footer* f = magic2footer(m, size);
  // contatore di allocazioni
  h->count = magic_counter++;
  // dimensione
  h->size = size;
  // linea
  h->line = line;
  // file
  h->file = file;
  // tag
  h->tag = f->tag = m;
  // riempie di spazzatura il campo dati
  magic_data_garbage_fill( m, h->size );
  // inserisce nella lista e imposta i crc
  magic_insert( m );
  return magic2ptr(m);
}

static void* magic_alloc(size_t size) {
  return magic_alloc( size, __FILE__, __LINE__ );
}

// informa che da questo punto deve essere fatto il controllo delle deallocazioni
static void magic_start() {
  magic_counter_start = magic_counter;
}

// stampa lo stato dell'heap
static void magic_log() {
  PRECONDITION( debug_log );
  bool all_deleted = true;
  bool good = true;
  if (magic_base) {
    magic_header* i = magic_base;
    do {
      if (!magic_test(header2magic(i))) {
	good = false;
	// se l'header e' corrotto non puo' continuare la scansione della lista
	if (!magic_header_test( i )) {
	  char buf[DEBUG_OUTPUT_MAX];
          magic_info_get( header2magic(i), buf, sizeof(buf) );
	  fprintf(debug_log, "%s\n", buf );
  	  break;
	}
      }
      if (i->count >= magic_counter_start || !magic_test(header2magic(i))) {
	char buf[DEBUG_OUTPUT_MAX];
        magic_info_get( header2magic(i), buf, sizeof(buf) );
	fprintf(debug_log, "%s\n", buf );
	all_deleted = false;
      }
      i = i->next;
    } while (i!=magic_base);
  } 
  if (good) 
    if (all_deleted)
      fprintf(debug_log, "-- heap good and free\n" );
    else
      fprintf(debug_log, "-- heap good\n" );
  else
    fprintf(debug_log, "-- heap bad\n" );
}

// --------------------------------------------------------------------------
// profile
// informazioni di profiling sulle allocazioni effettuate

#define DEBUG_PROFILE_MAX 32

// numero di allocazioni divise per dimensione 
// dimensione -> indice di memorizzazione
//   0 -> 0
//   1 -> 1
//   2,3 -> 2
//   4,5,6,7 -> 3
static unsigned profile_stat[DEBUG_PROFILE_MAX];
// spazio allocato
static unsigned profile_total = 0;
// numero di allocazioni
static unsigned profile_counter = 0;

// stampa le informazioni di profiling
static void profile_log() {
  PRECONDITION( debug_log );
  fprintf( debug_log, "-- allocation profile result:\n");
  for(unsigned i=0;i<DEBUG_PROFILE_MAX;++i)
    if (profile_stat[i]) { 
      int low = (1U << i) >> 1;
      int high = (1U << i)-1;
      fprintf( debug_log, "%8d/%8d bytes, %6d allocations\n", low,high,(int)profile_stat[i]);
    }
  fprintf( debug_log, "-- %d allocations for %d Kbytes\n", (int)profile_counter, (int)profile_total/1024 );
}

// Raccoglie le infomrazioni sulle allocazioni, da chiamre ad ogni allocazione
static void profile_allocate(unsigned size) {
  profile_total += size;
  ++profile_counter;
  int l = 0;
  while (size) {
    size >>= 1;
    ++l;
  }
  if (l<DEBUG_PROFILE_MAX)
    ++profile_stat[l];
}

// -------------------------------------------------------------------------
// new, delete

void* operator new(size_t size) {
  profile_allocate( size );
  void* ptr = magic_alloc( size );
  RUNTIME( ptr );
  return ptr;
}

void operator delete(void* p) {
  if (!p) return;
  CHECKPTR( p );
  magic_free( p );
}

void* operator new(size_t size, const char* file, unsigned line) {
  profile_allocate( size );
  void* ptr = magic_alloc( size, file, line );
  RUNTIME( ptr );
  return ptr;
}

void* operator new[](size_t size) {
  profile_allocate( size );
  void* ptr = magic_alloc( size );
  RUNTIME( ptr );
  return ptr;
}

void* operator new[](size_t size, const char* file, unsigned line) {
  profile_allocate( size );
  void* ptr = magic_alloc( size, file, line );
  RUNTIME( ptr );
  return ptr;
}

void operator delete[](void* p) {
  if (!p) return;
  CHECKPTR( p );
  magic_free( p );
}

// --------------------------------------------------------------------------
// pointer

static bool debug_ptr_test(const void* ptr) {
  if (!ptr) return false;
  if (!magic_ptr_test(ptr)) return false;
  return true;
}

static bool debug_heap_test() {
  return magic_test();
}

void _debug_assert_heap(const char* file, int line, const char* function) {
  if (!debug_heap_test()) {
    _debug_assert( "Heap bad", "", file, line, function);
  }
}

void _debug_assert_ptr(const void* ptr, const char* cond, const char* file, int line, const char* function) {
  if (!ptr)
    _debug_assert( "Null pointer", cond, file, line, function);
  if (!debug_ptr_test(ptr)) {
    char buf[DEBUG_OUTPUT_MAX];
    magic_ptr_info_get( ptr, buf, sizeof(buf) );
    fprintf( stderr, "%s\n", buf);
    _debug_assert( "Invalid pointer", cond, file, line, function);
  }
}

// --------------------------------------------------------------------------
// init, done

static void debug_heap_init() {
  magic_start();
}

static void debug_heap_done() {
  profile_log();
  magic_log();
}

#endif

// **************************************************************************
// Assert

#if __DEBUG >= __DEBUG_GAMMA

void _debug_assert(const char* msg, const char* cond, const char* file, int line, const char* function) {
#if defined( USE_SLANG )
  SLtt_set_cursor_visibility( true );
  SLsmg_reset_smg();
  SLang_reset_tty();
#endif
#if defined( USE_CURSES )
  leaveok( stdscr, false );
  endwin();
#endif

  fprintf(stderr,"%s:%d:in %s:",file, (int)line, function );
  fprintf(stderr,"%s %s\n", msg, cond);
  fprintf(stderr,"compiled %s %s\n", __DATE__, __TIME__ );
  if (errno) perror("errno:");

#if __DEBUG >= __DEBUG_ALPHA  
  debug_trace( file, line, function, msg, cond );
  TRACE( "aborting..." );
  // assicura di chiudere e aggiornare il file di debug
  debug_log_done();
#endif  

  abort();
}

#endif

// **************************************************************************
// Debug

#if __DEBUG >= __DEBUG_GAMMA

void debug_init() {
#if __DEBUG >= __DEBUG_ALPHA
  debug_log_init();
  debug_heap_init();
#endif
}

void debug_done() {
#if __DEBUG >= __DEBUG_ALPHA
  debug_heap_done();
  debug_log_done();
#endif
}

#endif
