#if !defined(__DEBUG_H)
#define __DEBUG_H

#include <stdlib.h>
#include <iostream.h>

#define ERROR(e) cerr << __FILE__ << ": " << __LINE__ << ": " << __FUNCTION__ << ": " << e << endl, exit(EXIT_FAILURE)

// --------------------------------------------------------------------------
// livelli di debug

// debug per versione release
//   niente
#define __DEBUG_RELEASE        0

// debug per versione gamma
//   attiva RUNTIME, ABSTRACT, NOTIMP
#define __DEBUG_GAMMA          1

// debug per versione beta
//   attiva PRECONDITION
#define __DEBUG_BETA           2

// debug per versione alpha
//   attiva CHECK, CHECKPTR
//   attiva controllo puntatori
#define __DEBUG_ALPHA          3

// --------------------------------------------------------------------------
// livello di debug di default

#if defined( NDEBUG )
#define __DEBUG __DEBUG_GAMMA
#elif !defined( __DEBUG )
#define __DEBUG __DEBUG_ALPHA
#endif

// --------------------------------------------------------------------------
// nop

#if !defined ( NOP )
#define NOP ((void)(0))
#endif

// --------------------------------------------------------------------------
// debug letter

// lettera identificativa della versione
#if __DEBUG == __DEBUG_RELEASE
#define DEBUG_LETTER " "
#elif __DEBUG == __DEBUG_GAMMA
#define DEBUG_LETTER ""
#elif __DEBUG == __DEBUG_BETA
#define DEBUG_LETTER ""
#elif __DEBUG == __DEBUG_ALPHA
#define DEBUG_LETTER ""
#else
#error __DEBUG ?
#endif

// --------------------------------------------------------------------------
// Gestione modalita di debug
// - attiva in GAMMA, BETA, ALPHA

#if __DEBUG >= __DEBUG_GAMMA

// attiva il debug, da chiamare subito dopo l'avvio
#define DEBUG_INIT() debug_init()
void debug_init();

// disattiva il debug, chiamare prima di terminare
#define DEBUG_DONE() debug_done()
void debug_done();

#else

#define DEBUG_INIT()           NOP
#define DEBUG_DONE()           NOP

#endif

// ---------------------------------------------------------------------------
// Gestore avanzato dell'heap
// - attivo solo in ALPHA

#if __DEBUG >= __DEBUG_ALPHA

// allocatori
void* operator new(size_t size);
void* operator new(size_t size, const char* file, unsigned line);
void operator delete(void* p);

void* operator new[](size_t size);
void* operator new[](size_t size, const char* file, unsigned line);
void operator delete[](void* p);

// allocatore con riferimento posizionale
#define NEW new(__FILE__,__LINE__)

#else

#define NEW new

#endif

// --------------------------------------------------------------------------
// Macro assert per uso interno
// - attiva in GAMMA, BETA, ALPHA

#if __DEBUG >= __DEBUG_GAMMA

void _debug_assert(const char* msg, const char* cond, const char* file, int line, const char* function);
void _debug_assert_ptr(const void* ptr, const char* cond, const char* file, int line, const char* function);
void _debug_assert_heap(const char* file, int line, const char* function);

#define ASSERT(msg,p) ((p) ? (void)0 : (void) _debug_assert( msg, #p, __FILE__, __LINE__, __FUNCTION__ ))

// assert msg
#define ASSERTABORT(msg) _debug_assert( msg, "", __FILE__, __LINE__, __FUNCTION__ )

// assert ptr
#define ASSERTPTR(p) _debug_assert_ptr( p, #p , __FILE__, __LINE__, __FUNCTION__ )

// assert heap
#define ASSERTHEAP() _debug_assert_heap( __FILE__, __LINE__, __FUNCTION__ )

#endif

// --------------------------------------------------------------------------
// Trace di informazioni
// - attivo solo in ALPHA

#if __DEBUG >= __DEBUG_ALPHA

void debug_trace(const char* file, unsigned line, const char* function);
void debug_trace(const char* file, unsigned line, const char* function, const char* a);
void debug_trace(const char* file, unsigned line, const char* function, int a);
void debug_trace(const char* file, unsigned line, const char* function, const char* a, const char* b);
void debug_trace(const char* file, unsigned line, const char* function, const char* a, const char* b, const char* c);
void debug_trace(const char* file, unsigned line, const char* function, const char* a, int b);
void debug_trace(const char* file, unsigned line, const char* function, const char* a, char b);
void debug_trace(const char* file, unsigned line, const char* function, const char* a, int b, const char* c);
void debug_trace(const char* file, unsigned line, const char* function, const char* a, int b, int c);

#define TRACE(a...)         debug_trace(__FILE__, __LINE__, __FUNCTION__, ##a)

#else

#define TRACE(a...)       NOP

#endif

// --------------------------------------------------------------------------
// Controlli 

// checkptr
#if __DEBUG >= __DEBUG_ALPHA
#define CHECKPTR( p ) ASSERTPTR( p )
#else
#define CHECKPTR( p ) NOP
#endif

// checkheap
#if __DEBUG >= __DEBUG_ALPHA
#define CHECKHEAP() ASSERTHEAP()
#else
#define CHECKHEAP() NOP
#endif

// check
#if __DEBUG >= __DEBUG_ALPHA
extern const char* MSG_CHECK;
#define CHECK( p ) ASSERT( MSG_CHECK, p )
#else
#define CHECK( p ) NOP
#endif

// notimp
#if __DEBUG >= __DEBUG_GAMMA
extern const char* MSG_NOTIMP;
#define NOTIMP ASSERTABORT( MSG_NOTIMP )
#else
#define NOTIMP NOP
#endif

// runtime
#if __DEBUG >= __DEBUG_GAMMA
extern const char* MSG_RUNTIME;
#define RUNTIME(p) ASSERT( MSG_RUNTIME, p)
#else
#define RUNTIME(p) NOP
#endif

// abstract
#if __DEBUG >= __DEBUG_GAMMA
extern const char* MSG_ABSTRACT;
#define ABSTRACT ASSERTABORT( MSG_ABSTRACT )
#else
#define ABSTRACT NOP
#endif

// precondition
#undef PRECONDITION
#if __DEBUG >= __DEBUG_BETA
extern const char* MSG_PRECONDITION;
#define PRECONDITION(p) ASSERT( MSG_PRECONDITION, p)
#else
#define PRECONDITION(p) NOP
#endif

#define POSTCONDITION PRECONDITION


#endif // __DEBUG_H

