// $Log: dstream.cc,v $
// Revision 1.2  1997/09/10 13:48:04  am
// Sequenza ENDL differenziata per MSDOS
//
// Revision 1.1  1997/09/10 13:37:49  am
// Initial revision
//

#if !defined( __MSDOS__ )
#include <config.h>
#endif

#include <limits.h>
#include <ctype.h>
#include <string.h>

#include <debug.h>

#include <dstream.h>

// ---------------------------------------------------------------------------

#if defined( __MSDOS__ )
#define ENDL "\r\n"
#else
#define ENDL "\n"
#endif

inline ocstream& operator<<(ocstream& os, const char* s) {
  while (*s) os.put(*s++);
  return os;
}

inline ocstream& operator<<(ocstream& os, char c) {
  os.put(c);
  return os;
}

// ---------------------------------------------------------------------------

// generatore di caratteri pseudocasuali
// in/out:
//   seed seme
// return:
//   valore casuale nei primi CHAR_BIT bit, altri bit a 0
unsigned rand_char(unsigned long& seed) {
  seed = seed * 0x015a4e35UL + 1;
  return (unsigned)((seed >> 8)) % (1 << CHAR_BIT);
}

// ---------------------------------------------------------------------------
// icstream

icstream::~icstream() {
}

// ---------------------------------------------------------------------------
// icfstream

icfstream::icfstream(istream& Ais) : is(Ais) {
  last = is.get();
}

char icfstream::get() {
  PRECONDITION( last != EOF );
  int r = last;
  last = is.get();
  return r;
}

bool icfstream::eof() {
  return last == EOF;
}

// ---------------------------------------------------------------------------
// ocstream

ocstream::~ocstream() {
}

// ---------------------------------------------------------------------------
// ocfstream

ocfstream::ocfstream(ostream& Aos) : os(Aos) {
}

void ocfstream::put(char c) {
  os.put(c);
}

void ocfstream::flush() {
  os.flush();
}

// ---------------------------------------------------------------------------
// ocwrapstream

ocwrapstream::ocwrapstream(ocstream& Aos) :
  os(Aos){
  // spazio iniziale 
  max = 128;
    
  line = new char[max];
  len = 0;
  line[len] = 0;
}

ocwrapstream::~ocwrapstream() {
  PRECONDITION( len == 0 );
  delete [] line;
}

void ocwrapstream::flush() {
  if (len) {
    os << line;
    len = 0;
    line[len] = 0;
  }
  os.flush();
}

void ocwrapstream::put(char c) {
  // se necessario rialloca line
  if (len + 2 >= max) {
    // raddoppia lo spazio disponibile
    unsigned Amax = 2 * max;
    char* Aline = new char[Amax];
    memcpy( Aline, line, len + 1 );
    delete [] line;
    line = Aline;
    max = Amax;
  }
  PRECONDITION( len + 2 < max );
  // aggiunge il carattere
  line[len] = c;
  ++len;
  line[len] = 0;
  // se ultimo carattere  spazio prova a spezzare se serve
  if (c==' ' && len>WRAP) {
    // parte dalla prima colonna non valida
    int i = WRAP;
    // cerca il primo spazio al'indietro 
    while (i>=0 && line[i] != ' ') --i;
    PRECONDITION( i != 0 );
    if (i<=0) {
      // tutta la linea, anche se sborda
      line[len-1] = 0;
      os << line << ENDL;
      line[0] = 0;
      len = 0;
    } else {
      line[i] = 0;
      // output della linea, spazio convertito in newline
      os << line << ENDL;
      // caratteri rimanenti
      memmove( line, line + i + 1, len - i );
      len = strlen( line );
    }
  }
}

// ---------------------------------------------------------------------------
// icunwrapstream

icunwrapstream::icunwrapstream(icstream& Ais) : is(Ais) {
  pred_space = true;
  full = false;
}

char icunwrapstream::get() {
  PRECONDITION( full );
  full = false;
  return c;
}

bool icunwrapstream::eof() {
  while (!full && !is.eof()) {
    c = is.get();
    if (isspace(c)) {
      if (!pred_space) {
	c = ' ';
	full = true;
      }
      pred_space = true;
    } else { 
      full = true;
      pred_space = false;
    }
  }
  return !full;	 
}

// ----------------------------------------------------------------------------
// icscramblestream

icscramblestream::icscramblestream(icstream& Ais) : is(Ais) {
  seed = 0;
}
  
char icscramblestream::get() {
  return (unsigned char)(((unsigned char)is.get()) ^ rand_char(seed));
}

bool icscramblestream::eof() {
  return is.eof();
}

// ----------------------------------------------------------------------------
// ocscramblestream

ocscramblestream::ocscramblestream(ocstream& Aos) : os(Aos) {
  seed = 0;
}
  
void ocscramblestream::put(char c) {
  os.put( (unsigned char)(((unsigned char)c) ^ rand_char(seed)) );
}

void ocscramblestream::flush() {
  os.flush();
}

// ---------------------------------------------------------------------------
// ibstream

// return:
//   true se il numero di bit virtuali richiesto  inferiore alla dimensione del 
//        carattere, ci  fondamentale per controlalre che i bit virtuali non
//        causino la creazione di caratteri virtuali non presenti nello stream 
//        di input
bool ibstream::no_wrap() { 
  PRECONDITION( eof() ); 
  return virtual_bit < CHAR_BIT; 
}

// in:
//   Ais stream dove estrearre i caratteri
ibstream::ibstream(icstream& Ais) : is(Ais) {
  virtual_bit = 0;
  nbit = 0;
  mbit = 0;
}

// out:
//   assicura nbit >= Anbit, eventualmente aggiungendo bit virtuali
void ibstream::look(unsigned Anbit) {
  look_novirtual( Anbit );
  if (nbit < Anbit) {
    virtual_bit += Anbit - nbit;
    nbit = Anbit;
  }
}

// out:
//   assicura nbit >= Anbit a meno di fine dell'input
void ibstream::look_novirtual(unsigned Anbit) {
  while (nbit < Anbit) {
    PRECONDITION( nbit + CHAR_BIT <= CHAR_BIT*sizeof(unsigned) );
    if (!is.eof()) {
      unsigned m = (unsigned char)is.get();
      m <<= nbit;
      mbit |= m;
      nbit += CHAR_BIT;
    } else {
      break;
    }
  }
}

// note:
//   vengono prodotti bit virtuali se lo stream non contiene abbastanza bit 
bool ibstream::get() {
  look(1);
  bool bit = (mbit & 1) != 0;
  mbit >>= 1;
  --nbit;
  return bit;
}

// legge nbit bit
// in:
//   nbit numerodi bit da leggere
// out:
//   mbit bit letti, il primo bit letto e' quello meno significativo
//        i bit di mbit non usati vengono posti a 0
// note:
//   vengono prodotti bit virtuali se lo stream non contiene abbastanza bit 
void ibstream::get(unsigned Anbit, unsigned& Ambit) {
  look( Anbit );
  Ambit = mbit & ((1 << Anbit)-1);
  mbit >>= Anbit;
  nbit -= Anbit;
}

// confronta la maschera specificata con il contenuto dello stream
// in:
//   Anbit numero di bit della maschera
//   Ambit maschera di bit, il bit meno significativo  il prossimo bit 
//         dello stream, non deve contenere pi di Anbit
// return:
//   true i bit corrispondono alla maschera specificata
// note:
//   i bit non vengono estratti dallo stream
//   non vengono generati bit virtuali
//   se Anbit==0 ritorna true
bool ibstream::peek(unsigned Anbit, unsigned Ambit) {
  if (!Anbit) return true;
  look_novirtual( Anbit );
  return (mbit & ((1 << Anbit)-1)) == Ambit;
}

// salta i prossimi Anbit
// note:
//   vengono prodotti bit virtuali se lo stream non contiene abbastanza bit 
void ibstream::skip(unsigned Anbit) {
  look( Anbit );
  mbit >>= Anbit;
  nbit -= Anbit;
}

// ---------------------------------------------------------------------------
// obstream

obstream::obstream(ocstream& Aos) : os(Aos) {
  nbit = 0;
  mbit = 0;
}

// inserisce un bit
// note:
//   un carattere viene effettivamento immesso nello stream di caratteri
//   SOLO quando  completato, ci significa che se il numero di bit TOTALE 
//   inserito non  esattamente multiplo del numero di bit per carattere
//   gli ultimi bit vengono persi
void obstream::put(bool Abit) {
  if (Abit)
    mbit |= 1 << nbit;
  ++nbit;
  PRECONDITION( nbit <= CHAR_BIT );
  if (nbit == CHAR_BIT) {
    unsigned m = mbit;
    char c = (unsigned char)m;
    os.put( c );
    nbit = 0;
    mbit = 0;
  }
}

// inserisce Anbit bit presi da Ambit
// in: 
//   Anbit numero di bit da scrivere
//   Ambit bit da scrivere, il primo bit scritto e' il meno significativo
void obstream::put(unsigned Anbit, unsigned Ambit) {
  for(unsigned i=0;i<Anbit;++i)
    put( (Ambit & (1 << i)) != 0 );
}

void obstream::flush() {
  os.flush();
}
