// $Log: iobit.h,v $
// Revision 1.5  1997/09/03 07:02:23  am
// Correzioni minori
//
// Revision 1.4  1997/09/02 21:32:53  am
// Corretto bug in word_wrap
// Modificato comportamento di ioconnect
// Aggiunti iochar2bit, siochar_nop, siobit_nop
//
// Revision 1.3  1997/09/01 20:23:55  am
// Corretti commenti
//

#if !defined( __IOBIT_H )
#define __IOBIT_H

#include <vector>

#include <debug.h>

#ifdef HEAP
#include <heap.h>
#endif

// ---------------------------------------------------------------------------
// ibit

// input di informazione binaria

class ibit {
  public:
// return:
//   estrae un bit
// precondition:  
//   empty() ==  false
// note:
//   non puo' fallire
virtual bool get() = 0;
// return:
//   true non c' emissione di bit
//   false c' emissione di bit  
virtual bool empty() const = 0;  
};

// ---------------------------------------------------------------------------
// obit

// output di informazione binaria

class obit {
  public:

// immette 1 bit  
// in:
//   bit bit da immettere
virtual void put(bool bit) = 0;
  
// informa che sono stati immessi tutti i bit
virtual void close() = 0;  
  };

// ---------------------------------------------------------------------------
// iobit

// flusso di informazione binaria

// precondition:
//   prima di chiamare put o close deve valere !empty

class iobit : public ibit, public obit {
  public:
    virtual ~iobit(); 
  };

// ---------------------------------------------------------------------------
// state

// classe con stato

// classe rappresentate lo stato
class state {
  public:
    virtual ~state();
  };

// classe con stato
class persistent {
  public:
  
// return:
//   puntatore alla classe contenente tutte le informazioni di stato,
// note:
//   la classe viene allocata dinamicamente e deve essere distrutta
virtual state* tell() const = 0;
  
// imposta lo stato specificato
// precodition:
//   lo stato specificato DEVE essere uno stato assunto in un istante di tempo
//   precedente dall'oggetto stesso
virtual void seek(const state*) = 0;
  };

// ---------------------------------------------------------------------------
// siobit

// flusso di informazione binaria con stato

class siobit : public iobit, public persistent {
  public:
  };

// ---------------------------------------------------------------------------
// siobit_nop

// flusso di informazione binaria con stato 
// NO OPERATION

class siobit_nop_state : public state { 
    bool b;
    bool full;
  
    friend class siobit_nop;
  public:
    virtual ~siobit_nop_state();
  };

class siobit_nop : public siobit {
    bool b;
    bool full;
  public:
    siobit_nop();
    virtual ~siobit_nop();
  
    virtual void put(bool b);
    virtual void close();
  
    virtual bool get();
    virtual bool empty() const;  
  
    virtual state* tell() const;
    virtual void seek(const state*);
  };


// ---------------------------------------------------------------------------
// ichar

// input di informazione a caratteri

class ichar {
  public:
    virtual char get() = 0;
    virtual bool empty() const = 0;  
};

// ---------------------------------------------------------------------------
// ochar

// output di informazione a carattere

class ochar {
  public:
    virtual void put(char ch) = 0;
    virtual void close() = 0;  
  };

// ---------------------------------------------------------------------------
// iochar

// flusso di informazione a caratteri
// precondition:
//   prima di chiamare put o close deve valere !empty

class iochar : public ichar, public ochar {
  public:
    virtual ~iochar(); 
  };

// ---------------------------------------------------------------------------
// siochar

// flusso di informazione a caratteri con stato

class siochar : public iochar, public persistent {
  public:
  };

// ---------------------------------------------------------------------------
// siochar_nop

// flusso di informazione a caratteri con stato 
// NO OPERATION

class siochar_nop_state : public state { 
    char c;
    bool full;
  
    friend class siochar_nop;
  public:
    virtual ~siochar_nop_state();
  };

class siochar_nop : public siochar {
    char c;
    bool full;
  public:
    siochar_nop();
    virtual ~siochar_nop();
  
    virtual void put(char ch);
    virtual void close();
  
    virtual char get();
    virtual bool empty() const;  
  
    virtual state* tell() const;
    virtual void seek(const state*);
  };

// --------------------------------------------------------------------------
// word wrap

// a capo automatico
// note:
//   le parole non vengono mai spezzate, puo' quindi accadere
//   di avere righe piu' lunghe del limite 
// note: 
//   c'e' emissione il piu' presto possibile 
// note:
//   gli spazi identificati con isspace() vengono convertiti in ' ' 
  
// massima lunghezza della parola
#define WORD_MAX 128
  
class word_wrap_state : public state {
    unsigned len;
    char map[WORD_MAX];
    unsigned input_begin;
    unsigned input_end;
    unsigned output_begin;
    unsigned output_end;
    bool first;  
  
    friend class word_wrap;
  public:
    virtual ~word_wrap_state();
  
#ifdef HEAP
    static allocator_fixed allocator;

    void* operator new(size_t size) {
      return allocator.allocate( size );
    }
    void operator delete(void* p) {
      allocator.deallocate( p );
    } 
#endif  
  
  };

class word_wrap : public siochar {
    // massimo numero di caratteri per riga (newline finale escluso)
    unsigned max;
    // caratteri gia' scritti sulla riga
    unsigned len;
    // buffer dei caratteri
    char map[WORD_MAX];
    // caratteri in input [input_begin,input_end)
    unsigned input_begin;
    unsigned input_end;
    // caratteri in output [output_begin,output_end)
    unsigned output_begin;
    unsigned output_end;

    // true se e' la prima parola della riga
    bool first;  
  
    void input_to_output(char c);
    void to_output(char c);
    void to_input(char c);
  
  public:
    word_wrap(unsigned Amax);
    virtual ~word_wrap();
  
    virtual void put(char ch);
    virtual void close();
  
    virtual char get();
    virtual bool empty() const;  
  
    virtual state* tell() const;
    virtual void seek(const state*);
  };
 
// ---------------------------------------------------------------------------
// char2bit

// conversione flusso da char a bit

class char2bit_state : public state {
    unsigned nbit;
    unsigned mbit;
    friend class char2bit;
  public:
    virtual ~char2bit_state();
  
#ifdef HEAP
    static allocator_fixed allocator;

    void* operator new(size_t size) {
      return allocator.allocate( size );
    }
    void operator delete(void* p) {
      allocator.deallocate( p );
    } 
#endif  

  };

class char2bit : public ibit, public ochar {
    unsigned nbit;
    unsigned mbit;
  public:
    char2bit();
    virtual ~char2bit();
    
    virtual void put(char ch);
    virtual void close();  
  
    virtual bool get();
    virtual bool empty() const;  

    virtual state* tell() const;
    virtual void seek(const state*);
  };

// ---------------------------------------------------------------------------
// bit2char

// conversione flusso da bit a char

class bit2char_state : public state {
    unsigned nbit;
    unsigned mbit;
    friend class bit2char;
  public:
    virtual ~bit2char_state();
  
#ifdef HEAP
    static allocator_fixed allocator;

    void* operator new(size_t size) {
      return allocator.allocate( size );
    }
    void operator delete(void* p) {
      allocator.deallocate( p );
    } 
#endif  
  
  };

class bit2char : public ichar, public obit {
    unsigned nbit;
    unsigned mbit;
  public:
    bit2char();
    virtual ~bit2char();
    
    virtual void put(bool bit);
    virtual void close();  
  
    virtual char get();
    virtual bool empty() const;  

    virtual state* tell() const;
    virtual void seek(const state*);
  };

// ---------------------------------------------------------------------------
// iochar2bit

// converte uno stream di char in uno stream di bit

class iochar2bit_state : public state {
    state* b2c;
    state* c;
    state* c2b;
    unsigned close_level;
  
    friend class iochar2bit;
  public:
    virtual ~iochar2bit_state();
  
#ifdef HEAP
    static allocator_fixed allocator;

    void* operator new(size_t size) {
      return allocator.allocate( size );
    }
    void operator delete(void* p) {
      allocator.deallocate( p );
    } 
#endif  

  };

class iochar2bit : public siobit {
    // conversione da bit a char
    bit2char* b2c;
    // flusso di char
    siochar* c;
    // conversione da char a bit
    char2bit* c2b;
    // livello di chiusura, 0=aperto, 1=primo chiuso, 2=secondo chiuso, ...
    unsigned close_level;

  public:
    iochar2bit(siochar* Ac);
    virtual ~iochar2bit();
  
    virtual void put(bool bit);
    virtual void close();
  
    virtual bool get();
    virtual bool empty() const;  

    virtual state* tell() const;
    virtual void seek(const state*);
  };

// ---------------------------------------------------------------------------
// ioconnect

// connette due flussi di informazioni binaria
//   input -> a -> b -> output

class ioconnect_state : public state {
    state* a;
    state* b;
    unsigned close_level;
  
    friend class ioconnect;
  public:
    virtual ~ioconnect_state();
  
#ifdef HEAP
    static allocator_fixed allocator;

    void* operator new(size_t size) {
      return allocator.allocate( size );
    }
    void operator delete(void* p) {
      allocator.deallocate( p );
    } 
#endif  

  };

class ioconnect : public siobit {
    siobit* a;
    siobit* b;
    // livello di chiusura, 0=aperto, 1=primo chiuso, 2=secondo chiuso, ...
    unsigned close_level;
  public:
    ioconnect(siobit* Aa, siobit* Ab);
    virtual ~ioconnect();
  
    virtual void put(bool bit);
    virtual void close();
  
    virtual bool get();
    virtual bool empty() const;  

    virtual state* tell() const;
    virtual void seek(const state*);
  };

// ---------------------------------------------------------------------------
// iobit_buffer_static

// bufferizza il flusso di informazioni binaria in input ed output 
// precondition:
//   lo stato e' valido (chiamata a tell()) solo a buffer vuoto (empty()==true)

class iobit_buffer_static : public siobit {
    vector<bool> data;
    unsigned output_pos;
  public:
    iobit_buffer_static();
    virtual ~iobit_buffer_static();
    
    virtual void put(bool bit);
    virtual void close();
  
    virtual bool get();
    virtual bool empty() const;  
  
    virtual state* tell() const;
    virtual void seek(const state*);
  
    unsigned size() const;
  };

// ---------------------------------------------------------------------------
// iobit_buffer

// bufferizza il flusso di informazioni binaria in input ed output 
// note:
//    empty rimane falso fino alla chiamata di close

class iobit_buffer_state : public state {
    unsigned output_pos;
    unsigned input_pos;
    friend class iobit_buffer;
  public:
    virtual ~iobit_buffer_state();

#ifdef HEAP
    static allocator_fixed allocator;

    void* operator new(size_t size) {
      return allocator.allocate( size );
    }
    void operator delete(void* p) {
      allocator.deallocate( p );
    } 
#endif  
  };

class iobit_buffer : public siobit {
    vector<bool> data;
    unsigned output_pos;
  public:
    iobit_buffer();
    virtual ~iobit_buffer();
    
    virtual void put(bool bit);
    virtual void close();
  
    virtual bool get();
    virtual bool empty() const;  
  
    virtual state* tell() const;
    virtual void seek(const state*);
  
    unsigned size() const;
  };


// ---------------------------------------------------------------------------
// end_protocol

// codifica il protocollo di fine file
//   string -> string 1  

class iobit_code_end_protocol : public iobit {
    // ultimo bit
    bool mbit;
    // mbit  valido
    bool nbit;
  public:
    iobit_code_end_protocol();
  
    virtual void put(bool bit);
    virtual void close();
  
    virtual bool get();
    virtual bool empty() const;  
  };

// decodifica il protocollo di fine file
//   string 1 0* -> string

class iobit_decode_end_protocol : public iobit {
    // bit bufferizzati in input
    bool ione; // bit 1 ricevuto
    unsigned izero; // numero di bit 0 ricevuti dopo il bit 1
    // bit bufferizzati in output
    bool oone;
    unsigned ozero;
  public:
    iobit_decode_end_protocol();
  
    virtual void put(bool bit);
    virtual void close();
  
    virtual bool get();
    virtual bool empty() const;  
  };

// ---------------------------------------------------------------------------
// decoder_standard

// decodificatore standard tramite funzione di hash

class decoder_standard_state : public state {
    bool sbit;
    unsigned nbit;
    friend class decoder_standard;
  public:
    virtual ~decoder_standard_state();
  
#ifdef HEAP  
    static allocator_fixed allocator;

    void* operator new(size_t size) {
      return allocator.allocate( size );
    }
    void operator delete(void* p) {
      allocator.deallocate( p );
    } 
#endif  
  };

class decoder_standard : public siobit {
    unsigned n;
    bool sbit;
    unsigned nbit;
  public:

// in:  
//   An numero di bit in input per avere un bit in output
decoder_standard(unsigned An);
  
    virtual ~decoder_standard();
  
    virtual void put(bool bit);
    virtual void close();
  
    virtual bool get();
    virtual bool empty() const;  
  
    virtual state* tell() const;
    virtual void seek(const state*);
  };

#endif

