// $Log: mimic.cc,v $
// Revision 1.8  1997/09/10 09:25:45  am
// Corrette messaggi
//
// Revision 1.7  1997/09/03 07:12:32  am
// Nuova gestione a stream di caratteri
//
// Revision 1.6  1997/09/02 21:34:21  am
// put e close non gestiscono piu' la condizione di errore
//
// Revision 1.5  1997/08/26 15:59:02  am
// Inclusione condizionale di config.h
//
// Revision 1.4  1997/06/12 20:17:01  am
// aggiunte minori ai commenti
//
// Revision 1.3  1997/06/12 19:52:57  am
// aggiunti switch STEGO_USE_TREE e CODER_USE_TREE
// aggiunte minori ai commenti
//
// Revision 1.2  1997/06/05 20:15:53  am
// add NOTREE switch
//
// Revision 1.1  1997/06/05 20:14:32  am
// Initial revision
//

#pragma implementation

#if !defined( __MSDOS__ )
#include <config.h>
#endif

#include <vector>
#include <list>

#include <debug.h>

#include <mimic.h>

// const_cast
template <class T> inline T& noconst(const T& A) { return (T&)A; }

// ----------------------------------------------------------------------------
// limite per cui accettare una probabilit rispetto a quella desiderata
//   | p - p_desiderata | < P_EPSILON
#define P_EPSILON 0.2

// ---------------------------------------------------------------------------
// symbol

// minima dimensione (esclusa) nel caso sia dispari per cui e' 
// possibile spezzare in due mantenendo la probabilit di circa 50% e 50% a 
// meno di P_EPSILON
#define SYMBOL_EQ_MIN (unsigned)( 1.0/(2.0*P_EPSILON) )

// sceglie in modo equiprobabile tra un insieme di size simboli un simbolo 
// usando i bit specificati
// in:
//   size numero di simboli
//   nbit numero di bit disponibili
//   mbit bit disponibili, supposti bit casuali distribuiti uniformemente
// return:
//   0<=return_value<size return_value e' il simbolo scelto
//   size non ci sono abbastanza bit per scegliere il simbolo
// note:
//   nbit non deve contenere piu' bit del necessario, cio' significa 
//   che bisogna chiamare in sequenza con nbit 0,1,2,3,...
unsigned symbol_eq_get(unsigned size, unsigned nbit, unsigned mbit) {
  PRECONDITION( size > 0 );
  unsigned begin = 0;
  unsigned end = size;
  unsigned i = 0;
  for(;begin + 1 != end && i<nbit;++i) {
    // se dispari e troppo corto
    if ((end - begin) % 2 != 0 && (end - begin) <= SYMBOL_EQ_MIN) {
      // scelto casualmente
      begin = begin + rand() % (end - begin);
      end = begin + 1;
    } else {
      // sceglie una delle due meta'
      unsigned middle = (begin + end) / 2;
      if ((mbit & (1 << i)) == 0) {
        end = middle;
      } else {
        begin = middle;
      }
    }
  }
  PRECONDITION( i == nbit ); // sono stati forniti troppi bit
  // se univoco
  if (begin + 1 == end) {
    return begin;
  }
  // non ci sono abbastanza bit
  return size;
}

// operazione inversa della precedente, dato il simbolo ne ritorna i bit
// che ne hanno deciso la scelta
// in:
//   n simbolo
//   size numero dei simboli
// out:
//   nbit numero di bit
//   mbit bit
void symbol_eq_put(unsigned n, unsigned size, unsigned& nbit, unsigned& mbit) {
  nbit = 0;
  mbit = 0;
  unsigned begin = 0;
  unsigned end = size;
  while (begin + 1 != end) {
    // se dispari e troppo corto
    if ((end - begin) % 2 != 0 && (end - begin) <= SYMBOL_EQ_MIN) {
      end = begin + 1;
    } else {
      // sceglie una delle due meta'
      unsigned middle = (begin + end) / 2;
      if (n<middle) {
        end = middle;
      } else {
        begin = middle;
        mbit |= 1 << nbit;
      }
      ++nbit;
    }
  }
}

// ---------------------------------------------------------------------------
// element

element::element() {
  index = 0;
  presence = 0;
  nbit = 0;
  mbit = 0;
}

element::element(const element& A) {
  index = A.index;
  presence = A.presence;
  nbit = A.nbit;
  mbit = A.mbit;
}

element::element(element_index Aindex) {
  index = Aindex;
  presence = 0;
  nbit = 0;
  mbit = 0;
}

ostream& operator<<(ostream& os, const element& A) {
  os << (unsigned)A.index << " " << A.presence;
  return os;
}

istream& operator>>(istream& is, element& A) {
  unsigned index;
  is >> index;
  A.index = index;
  is >> A.presence;
  if (A.presence<1) {
    ERROR("input error");
  }
  A.nbit = 0;
  A.mbit = 0;
  
  return is;
}

// ----------------------------------------------------------------------------

// albero binario per l'assegnamento dei codici di partizione e creazione dell'albero
// note:
//   l'inserimento avviene in modo che sia bilanciato per la somma dei
//   valori presence_get() degli elementi

class partition_tree {
    // rami figli
    partition_tree* bit0;
    partition_tree* bit1;
    // elemento
    element* data;
    // invariant: (data==0) == (bit0!=0) == (bit1!=0)
  public:
    partition_tree();
    ~partition_tree();

    void insert(element* A);
    unsigned presence_get() const;

    void bit_create(unsigned Anbit, unsigned Ambit);
    void bit_set(unsigned Anbit, unsigned Ambit);
  
    context_tree* tree_create();
    void tree_insert( context_tree* tree );
  };

partition_tree::partition_tree() {
  bit0 = 0;
  bit1 = 0;
  data = 0;
}

partition_tree::~partition_tree() {
  if (bit0) delete bit0;
  if (bit1) delete bit1;
}

// somma delle presenze dei sottorami
unsigned partition_tree::presence_get() const {
  if (!bit0) {
    PRECONDITION( !bit0 && !bit1 );
    if (data)
      return data->presence_get();
    else
     return 0;
  } else {
    PRECONDITION( bit0 && bit1 && !data);
    return bit0->presence_get() + bit1->presence_get();
  }
}

// inserisce una maschera
void partition_tree::insert(element* A) {
  if (!bit0) {
    PRECONDITION( !bit0 && !bit1 );
    if (!data)
      data = A;
    else {
      bit0 = new partition_tree;
      bit1 = new partition_tree;
      bit0->insert( data );
      bit1->insert( A );
      data = 0;
    }
  } else {
    PRECONDITION( bit0 && bit1 && !data);
    // la inserisce nel ramo con meno presenze
    if (bit0->presence_get() < bit1->presence_get())
      bit0->insert( A );
    else
      bit1->insert( A );
  }
}

// crea i codici per tutto il sottoalbero
// note:
//   Il codice viene creato solo se la partizione  bilanciata,
//    cio la probabilit dei rami non si discosti dal 50% pi di P_EPSILON
void partition_tree::bit_create(unsigned Anbit, unsigned Ambit) {
  if (data) {
    PRECONDITION( !bit0 && !bit1 && data);
    data->bit_set( Anbit, Ambit );
  } else {
    PRECONDITION( bit0 && bit1 && !data);
    // contatori di presenze
    unsigned pl = bit0->presence_get();
    unsigned pr = bit1->presence_get();
    // probabilit
    double p = (double)pl / (double)(pl+pr);
    // se la probabilit non si discosta troppo da quella effettiva
    if (0.5-P_EPSILON <= p && p <= 0.5+P_EPSILON) {
      // genera due nuovi codici per i sottorami
      bit0->bit_create( Anbit + 1, Ambit);
      bit1->bit_create( Anbit + 1, Ambit | (1 << Anbit) );
    } else {
      // se partizione troppo sbilanciata interrompe la generazione del codice
      bit0->bit_set( Anbit, Ambit );
      bit1->bit_set( Anbit, Ambit );
    }
  }
}

// crea l'albero
context_tree* partition_tree::tree_create() {
  if (data) {
    PRECONDITION( !bit0 && !bit1 );
    context_tree* tree = new context_tree();
    tree->insert( data );
    return tree;
  } else {
    PRECONDITION( bit0 && bit1 );
    // contatori di presenze
    unsigned pl = bit0->presence_get();
    unsigned pr = bit1->presence_get();
    // probabilit
    double p = (double)pl / (double)(pl+pr);
    // se la probabilit non si discosta troppo da quella effettiva
    if (0.5-P_EPSILON <= p && p <= 0.5+P_EPSILON) {
      // crea i due sottoalberi
      return new context_tree( bit0->tree_create(), bit1->tree_create() );
    } else {
      context_tree* tree = new context_tree;
      bit0->tree_insert( tree );
      bit1->tree_insert( tree );
      return tree;
    }
  }
}

// inserisce tutti gli elementi dell'albero nello stesso nodo dell'albero
void partition_tree::tree_insert( context_tree* tree ) {
  PRECONDITION( tree );
  if (data) {
    PRECONDITION( !bit0 && !bit1 );
    tree->insert( data );
  } else {
    PRECONDITION( bit0 && bit1 );
    bit0->tree_insert( tree );
    bit1->tree_insert( tree );
  }
}

// imposta i codici fissi per tutto il sottoalbero
void partition_tree::bit_set(unsigned Anbit, unsigned Ambit) {
  if (data) {
    PRECONDITION( !bit0 && !bit1 && data );
    data->bit_set( Anbit, Ambit );
  } else {
    PRECONDITION( bit0 && bit1 && !data );
    bit0->bit_set( Anbit, Ambit );
    bit1->bit_set( Anbit, Ambit );
  }
}

// confronto tra le presenze
int element_presence_cmp(element* r1, element* r2) {
  return r1->presence_get() > r2->presence_get();
}

// restituisce l'albero di partizione
// in:
//   tab elementi da inserire nell'albero
// out:
//   tab svuotato
// return:
//   puntatore all'albero
// note:
//   Viene generato l'albero avendo l'accortezza di bilanciarlo 
//   sulla frequenza dei simboli.  
//   Cio arrivare approssimativamente alla situazione 50% da una parte 
//   e 50% dall'altra 
partition_tree* make_partition(element_set::const_iterator begin, element_set::const_iterator end, unsigned reserve) {
  vector<element*> tab;
  // riserva lo spazio necessario
  tab.reserve( reserve );
  // inserisce gli elementi
  for(element_set::const_iterator i = begin;i != end; ++i)
    tab.push_back( &noconst(*i) );
  // euristica knapsack, inserisce prima gli elementi pi grossi
  sort( tab.begin(), tab.end(), pointer_to_binary_function<element*,element*,int>(element_presence_cmp) );
  partition_tree* tree = new partition_tree;
  for(vector<element*>::iterator j=tab.begin();j!=tab.end();++j) 
    tree->insert( *j );
  tab.erase( tab.begin(), tab.end() );
  return tree;
}

// ----------------------------------------------------------------------------
// context

unsigned context::n = 0;

context::context() {
}

context::context(const context& A) {
  for(unsigned i=0;i<n_get();++i)
    pred[i] = A.pred[i];
}

context& context::operator=(const context& A) {
  for(unsigned i=0;i<n_get();++i)
    pred[i] = A.pred[i];
  return *this;
}

// inserisce un nuovo elemento nel constesto
void context::index_shift(element_index index) {
  for(unsigned i=n_get()-1;i>0;--i)
    pred[i] = pred[i-1];
  pred[0] = index;
}

bool context::operator==(const context& A) const {
  for(unsigned i=0;i<n_get();++i)
    if (pred[i] != A.pred[i]) return false;
  return true;
}

bool context::operator<(const context& A) const {
  for(unsigned i=0;i<n_get();++i) {
    if (pred[i] < A.pred[i]) return true;
    if (pred[i] > A.pred[i]) return false;
  }
  return false;
}

// ----------------------------------------------------------------------------

#ifdef HEAP
allocator_fixed context_tree::allocator( sizeof( context_tree ), allocator_page );
#endif

context_tree::context_tree(context_tree* Abit0, context_tree* Abit1) {
  bit0 = Abit0;
  bit1 = Abit1;
}

context_tree::context_tree() {
  bit0 = 0;
  bit1 = 0;
}

context_tree::~context_tree() {
  if (bit0) delete bit0;
  if (bit1) delete bit1;
}

void context_tree::insert(const element* A) {
  PRECONDITION( !bit0 && !bit1 );
  // aggiunge spazio per 1 elemento
  if (data.capacity() <= data.size())
    data.reserve( data.size() + 1 );
  // aggiunge elemento
  data.push_back( A );
}

// ----------------------------------------------------------------------------
// context_next

context_next::context_next() {
  tree = 0;
}

context_next::~context_next() {
  if (tree) {
    delete tree;
  }
}

context_next::context_next(const context_next& A) : context( A ), data(A.data) {
  tree = 0;
}

context_next::context_next(const context& A) : context( A ) {
  tree = 0;
}

context_next& context_next::operator=(const context_next& A) {
  // cancella l'albero
  if (tree) delete tree;
  tree = 0;
  
  context::operator=(A);
  data = A.data;
  return *this;
}

ostream& operator<<(ostream& os, const context_next& A) {
  for(unsigned i=0;i<context_next::n_get();++i) {
    if (i) os << " ";
    os << (unsigned)A.pred[i];
  }
  os << endl;
  os << A.data.size() << endl;
  for(element_set::const_iterator i = A.data.begin(); i != A.data.end(); ++i)
    os << *i << endl;
  return os;
}

istream& operator>>(istream& is, context_next& A) {
  // cancella l'albero
  if (A.tree) delete A.tree;
  A.tree = 0;
  
  for(unsigned i=0;i<context::n_get();++i) {
    unsigned index;
    is >> index;
    A.pred[i] = index;
  }
  is.ignore(2,'\n');
  unsigned size;
  is >> size;
  is.ignore(2,'\n');
#ifdef ELEMENT_SET_IS_SET
  for(unsigned i=0;i<size;++i) {
    element e;
    is >> e;
    A.data.insert( e );
    is.ignore(2,'\n');
  }
#else  
  // riserva lo spazio
  A.data.reserve( size );
  for(unsigned i=0;i<size;++i) {
    element e;
    is >> e;
    // inserisce in coda, sono gi ordinati
    A.data.push_back( e );
    is.ignore(2,'\n');
  }
#endif
  return is;
}

// imposta i codici di partizione
// i codici vengono impostati cercando di aprossimare la probabilit
// degli elementi, ci comporta codici non univoci
void context_next::partition_bit_set() {
  // cancella l'albero
  if (tree) delete tree;
  tree = 0;
  
  if (data.size())  {
    // partiziona
    partition_tree* pt = make_partition( data.begin(), data.end(), data.size() );
    // imposta i codici
    pt->bit_create(0,0);
    delete pt;
  } 
}

// imposta l'albero di partizione
// i codici vengono impostati cercando di aprossimare la probabilit
// degli elementi, ci comporta codici non univoci
void context_next::partition_tree_set() {
  // cancella l'albero
  if (tree) delete tree;
  tree = 0;

  if (data.size())  {
    // partiziona
    partition_tree* pt = make_partition( data.begin(), data.end(), data.size() );
    // crea l'albero
    tree = pt->tree_create();
    delete pt;
  } 
}

unsigned context_next::presence_get() const {
  unsigned presence = 0;
  for(element_set::const_iterator i = data.begin();i!=data.end();++i)
    presence += (*i).presence_get();
  return presence;
}

void context_next::presence_reset() {
  for(element_set::iterator i = data.begin();i!=data.end();++i)
    noconst(*i).presence_reset();
}

// ----------------------------------------------------------------------------
// mimic

mimic::mimic() {
}

// conta il numero di maschere
unsigned mimic::element_count() const {
  unsigned n = 0;
  context_next_set::const_iterator i = begin();
  while (i!=end()) {
    n += (*i).size();
    ++i;
  }
  return n;
}

// conta il numero di gruppi
unsigned mimic::context_count() const {
  return data.size();
}

// calcola il bit rate medio [bit/simbolo]
double mimic::bit_rate() const {
  context_next_set::const_iterator it = begin();
  double rate = 0;
  double presence = 0;
  while (it != end()) {
    element_set::const_iterator git = (*it).begin();
    while (git != (*it).end()) {
      presence += (*git).presence_get();
      rate += (double)(*git).nbit_get() * (*git).presence_get();
      ++git;
    }
    ++it;
  }
  return rate / presence;
}

void mimic::partition_bit_set() {
  for(context_next_set::iterator i = begin();i != end();++i)
    noconst(*i).partition_bit_set();
}

void mimic::partition_tree_set() {
  for(context_next_set::iterator i = begin();i != end();++i)
    noconst(*i).partition_tree_set();
}

ostream& operator<<(ostream& os, const mimic& A) {
  os << A.data.size() << endl;
  for(context_next_set::const_iterator i = A.begin(); i != A.end(); ++i) 
    os << *i;
  return os;
}

istream& operator>>(istream& is, mimic& A) {
  // cancella il contenuto attuale
  A.data.erase( A.data.begin(), A.data.end() );

  unsigned size;
  is >> size;
#ifdef CONTEXT_SET_IS_SET  
  for(unsigned i=0;i<size;++i) {
    context_next c;
    is >> c;
    A.data.insert( c );
  }
#else
  for(unsigned i=0;i<size;++i) {
    context_next c;
    is >> c;
    A.data.push_back( c );
  }
#endif
  
  // imposta i codici
  A.partition_bit_set();
  
  return is; 
}

// ---------------------------------------------------------------------------
// import

void mimic::import(iestream& is) {
  // cancella il contenuto attuale
  data.erase( data.begin(), data.end());

  // contesto
  context current;

  // importa il contesto iniziale
  for(unsigned i=0;i<context::n_get();++i) {
    element_index index = is.get();
    current.index_shift( index );
  }
  
  while (!is.eof()) {
    element_index index = is.get();  
    // inserisce il contesto se non c' gi
    pair<context_next_set::iterator,bool> i = insert( context_next( current ) );
    // elemento da inserire
    element e( index );
    e.presence_reset();
    e.presence_inc();
    // inserisce
    pair<element_set::iterator,bool> j = noconst(*i.first).insert( e );
    if (!j.second) {
      // l'elemento c'era gi, incrementa le presenze
      noconst(*j.first).presence_inc();
    } 
    // nuovo contesto
    current.index_shift( index );
  }

  // elimina rami morti
  shrink();

  if (!data.size()) {
    ERROR( "dictionary empty" );
  }

  // imposta i codici
  partition_bit_set();

}

// elimina dal dizionario tutti i rami morti
// note:
//   vengono eliminati tutti gli elementi che generano contesti inesistenti
//   vengono eliminati tutti i contesti che non hanno elementi successori
void mimic::shrink() {
  for(context_next_set::iterator i=begin();i!=end();) {
    for(element_set::iterator j=noconst(*i).begin();j!=noconst(*i).end();) {
      context current = *i;
      current.index_shift( (*j).index_get() );
      context_next_set::const_iterator k = find( context_next( current ) );
      // se non trova il contesto cancella l'elemento che lo genera
      if (k==end()) {
	// cancella l'elemento
        noconst(*i).erase( j );
        // ricomincia dall'inizio, serve per rivalidare l'iteratore j
        j = noconst(*i).begin();
      } else
        ++j;
    }
    // se il contesto non ha successori
    if (!(*i).size()) {
      // cancella il contesto
      data.erase( i );
      // ricomincia dall'inizio, serve per rivalidare l'iteratore i e
      // per propagare l'effetto della modifica su tutti i contesti
      i = begin();
    } else
      ++i;
  }
}

// ---------------------------------------------------------------------------
// operazioni stego/unstego sul seme

// seeds  un vettore di puntatori ai contesti NEL dizionario
// da utilizzare come semi iniziali

// seme dato il codice
// return:
//   iteratore che punta al seme interno a seeds
//   seeds.end() non bastano i bit
vector<context_next*>::const_iterator mimic::seed_get(const vector<context_next*>& seeds, unsigned nbit, unsigned mbit) const {
  PRECONDITION( seeds.size() > 0 );
  unsigned i = symbol_eq_get( seeds.size(), nbit, mbit );
  if (i == seeds.size())
    return seeds.end();
  else
    return seeds.begin() + i;
}

// codice dato il seme
void mimic::seed_put(const context& seed, const vector<context_next*>& seeds, unsigned& nbit, unsigned& mbit) const {
  PRECONDITION( seeds.size() > 0 );
  // cerca il seme
  vector<context_next*>::const_iterator i = seeds.begin();
  while (i != seeds.end() && seed != **i) 
    ++i;
  if (i == seeds.end() ) {
    ERROR( "dictionary error, unknow seed" );
  }
  symbol_eq_put( i - seeds.begin(), seeds.size(), nbit, mbit );
}

// sceglie il seme codificando informazione dallo stream
context mimic::seed_stego(const vector<context_next*>& seeds, ibstream& is) const {
  PRECONDITION( seeds.size() > 0 );
  // legge i bit
  unsigned nbit = 0;
  unsigned mbit = 0;
  vector<context_next*>::const_iterator i = seed_get( seeds, nbit, mbit );
  while (i == seeds.end()) { 
    if (is.get())
      mbit |= 1 << nbit;
    ++nbit;
    i = seed_get( seeds, nbit, mbit );
  } 
  return **i;
}

// dato il seme restituisce l'informazione nello stream
void mimic::seed_unstego(const context& seed, const vector<context_next*>& seeds, obstream& os) const {
  PRECONDITION( seeds.size() > 0 );
  unsigned nbit;
  unsigned mbit;
  seed_put(seed, seeds, nbit, mbit );
  os.put( nbit, mbit );
}


// ---------------------------------------------------------------------------
// operazioni stego/unstego 

// assegnamento dei codici ai simboli fisso

//#define STEGO_USE_TREE
//   Utilizza gli alberi, piu' memoria richiesta ma piu' veloce
//   complessita': per ogni simbolo in output e' pari al numero di elementi 
//   nel contesto
#undef STEGO_USE_TREE
//   Non utilizza gli alberi, meno memoria richiesta ma piu' lento
//   complessita': per ogni simbolo in output e' pari al numero di elementi 
//   nel contesto con il codice corretto

// note:
//   richiedono i bit gia' impostati (chiamata a bit_partition_set)
// note:
//   l'unica modifica approtata allo stato e' la eventuale creazione degli 
//   alberi di partizione

// in:
//   seed seme con cui iniziare la creazione
void mimic::stego(const vector<context_next*>& seeds, oestream& os, ibstream& is) {
#if !defined( STEGO_USE_TREE )
  
  // contesto corrente
  context current = seed_stego( seeds, is );
  
  // output del seme
  for(unsigned i=0;i<context::n_get();++i) {
    element_index index = current.index_forward_get( i );
    os.put( index );
  }

  while (!is.eof()) {
    // cerca il contesto
    context_next_set::const_iterator i = find( context_next( current ) );
    if (i == end()) {
      ERROR("dictionary error, unknow context");
    }
    // per ogni elemento del gruppo cerca quelli con il codice appropriato
    vector<const element*> tab;
    // presuppone che nella maggior parte dei casi non si superi la
    // dimensione di size()/2 elementi
    tab.reserve( (*i).size() / 2 );
    // contatore di presenze totali
    unsigned presence = 0;
    // esami i possibili contesti successivi
    element_set::const_iterator k = (*i).begin();
    while (k != (*i).end()) {
      // se il contesto ha la maschera di bit corretta
      if (is.peek( (*k).nbit_get(), (*k).mbit_get())) {
	// inserisce il contesto nell' insieme dei contesti validi
        tab.push_back( (const element*)&*k );
        // incrementa le presenze
        presence += (*k).presence_get();
      }
      ++k;
    }
    // se non ci sono contesti validi 
    if (tab.size() == 0) {
      ERROR( "dictionary error, no next" );
    }
    // elemento scelto
    vector<const element*>::iterator j = tab.begin();
    // se gli elemento sono pi di 1 se ne sceglie uno a caso
    if (tab.size() > 1) {
      int r = rand() % presence;
      for(;r >= (int)(*j)->presence_get();++j)
        r -= (*j)->presence_get();
    } 
    // salta i bit dello stream usati per la selezione del contesto
    is.skip( (*j)->nbit_get() );
    // output dell'elemento
    os.put( (*j)->index_get() );
    // nuovo contesto
    current.index_shift( (*j)->index_get() );
  }
  if (!is.no_wrap()) {
    ERROR( "character wrap" );
  }
#else

  // imposta l'albero
  partition_tree_set();
  
  // contesto corrente
  context current = seed_stego( seeds, is );
  
  // output del seme
  for(unsigned i=0;i<context::n_get();++i) {
    element_index index = current.index_forward_get( i );
    os.put( index );
  }

  while (!is.eof()) {
    // cerca il contesto
    context_next_set::const_iterator i = find( context_next( current ) );
    if (i == end()) {
      ERROR("dictionary error, unknow context");
    }
    if ((*i).size() == 0) {
      ERROR( "dictionary error, no next" );
    }
    // albero del contesto corrente
    const context_tree* tree = (*i).tree_get();
    // legge in input fino a raggiungere la foglia dell'albero
    while (!tree->leaf()) {
      tree = tree->put( is.get() );
    }
    // elemento scelto
    vector<const element*>::const_iterator j = tree->begin();
    // se gli elemento sono pi di 1 se ne sceglie uno a caso
    if (tree->size() > 1) {
      // calcola il contatore di presenze totali
      vector<const element*>::const_iterator k = tree->begin();
      unsigned presence = 0;
      while (k != tree->end()) {
	presence += (*k)->presence_get();
	++k;
      }
      // estrae a caso
      int r = rand() % presence;
      for(;r >= (int)(*j)->presence_get();++j)
        r -= (*j)->presence_get();
    } 
    // output dell'elemento scelto
    os.put( (*j)->index_get() );
    // nuovo contesto
    current.index_shift( (*j)->index_get() );
  }
  if (!is.no_wrap()) {
    ERROR( "character wrap" );
  }
#endif

  os.flush();
}

void mimic::unstego(const vector<context_next*>& seeds, obstream& os, iestream& is) {
  // contesto corrente
  context current;
  
  // legge il contesto
  for(unsigned i=0;i<context::n_get();++i) 
    current.index_shift( is.get() );
  
  // informazione del seme
  seed_unstego( current, seeds, os );
  
  while ( !is.eof() ) {
    // prossimo elemento
    element_index c = is.get();
    // cerca il gruppo
    context_next_set::const_iterator i = find( context_next( current ) );
    if (i == end()) {
      ERROR( "dictionary error, unknow context" );
    }
    element_set::const_iterator j = (*i).find( c );
    if (j == (*i).end()) {
      ERROR( "dictionary error, unknow element");
    }
    // scrive i bit associati al carattere
    os.put( (*j).nbit_get(), (*j).mbit_get() );
    // nuovo contesto
    current.index_shift( c );
  }	   
  
  os.flush();
}

// ---------------------------------------------------------------------------
// alpha

alpha::alpha() { 
}

alpha::~alpha() {
  erase();
}

double alpha::bit_length() const {
  double r = 0;
  for(alpha_index::const_iterator i = mindex.begin();i!=mindex.end();++i) 
    r += CHAR_BIT * strlen( *i );
  r /= size();
  return r;
}

void alpha::erase() {
  // cancella mstring
  mstring.erase( mstring.begin(), mstring.end() );
  // dealloca le stringhe
  for(alpha_index::iterator i=mindex.begin();i!=mindex.end();++i) 
    delete [] *i;
  // cancella mindex
  mindex.erase( mindex.begin(), mindex.end() );
}

element_index alpha::insert(const char* A) {
  alpha_string::const_iterator i = mstring.find( (char*)A );
  if (i == mstring.end()) {
    // se non c' inserisce
    unsigned l = strlen( A );
    char* s = new char[l+1];
    strcpy(s,A);
    alpha_string::value_type e(s,mindex.size() );
    mstring.insert( e );
    mindex.push_back( e.first );
    return e.second;
  } else
    return (*i).second;
}

ostream& operator<<(ostream& os, const alpha& A) {
  os << A.size() << endl;
  for(alpha_index::const_iterator i = A.mindex.begin();i!=A.mindex.end();++i) 
    os << *i << endl;
  return os;
}

char* gets(istream& is) { 
  unsigned mac = 0;
  unsigned max = 64;
  char* s = new char[max];
  int c = is.get();
  while (!is.bad() && c!=EOF && c!='\n') {
    // rialloca
    if (mac + 1 >= max) {
      unsigned Amax = max * 2;
      char* As = new char[Amax];
      memcpy( As, s, mac );
      delete [] s;
      s = As;
      max = Amax;
    }
    s[mac] = c;
    ++mac;
    c = is.get();
  }
  s[mac] = 0;
  return s;
}


istream& operator>>(istream& is, alpha& A) {
  A.erase();
  unsigned n;
  is >> n;
  is.ignore(2,'\n');
  // riserva memoria per il vettore
  A.mindex.reserve( n );
  for(unsigned i=0;i<n;++i) {
    char* line = gets( is );
    A.insert( line );
    delete [] line;
  }
  return is;
}
     
// ---------------------------------------------------------------------------
// dictionary

dictionary::dictionary() {
}
      
dictionary::~dictionary() {  
}
      
// return:
//   numero di elementi nel dizionario, cio la somma totale
//   dei successori di ogni contesto
unsigned dictionary::element_count() const { 
  return diz.element_count(); 
}

// return:
//   numero di contesti nel dizionario
unsigned dictionary::context_count() const { 
  return diz.context_count(); 
}

// return:
//   numero di elmenti dell'alfabeto
unsigned dictionary::alpha_count() const { 
  return alp.size(); 
}

// return:
//   fattore di espansione APPROSSIMAIVO, non tiene conto della
//   frequenza degli elementi/contesti
double dictionary::expansion_factor() const { 
  return alp.bit_length() / diz.bit_rate(); 
}
  
ostream& operator<<(ostream& os, const dictionary& A) {
  os << context::n_get() << endl;
  os << A.alp;
  os << A.diz;
  return os;
}

istream& operator>>(istream& is, dictionary& A) {
  unsigned n;
  is >> n;
  if (n<CONTEXT_MIN || n>CONTEXT_MAX) {
    ERROR( "input error" );
  }
  context::n_set( n );
  is.ignore(2,'\n');
  is >> A.alp;
  is >> A.diz;
  A.make_seeds();
  
  return is;
}

// ---------------------------------------------------------------------------
// coder_dictionary

#ifdef HEAP
allocator_fixed coder_dictionary_state::allocator( sizeof(coder_dictionary_state), allocator_page );
#endif

coder_dictionary_state::coder_dictionary_state(state* Abuffer) {
  buffer = Abuffer;
}

coder_dictionary_state::~coder_dictionary_state() {
  if (buffer) delete buffer;
}

#define CODER_USE_TREE
//   Utilizza gli alberi, piu' memoria richiesta ma piu' veloce
//   complessita': per ogni bit in input e' pari al numero di elementi 
//   nel contesto
//#undef CODER_USE_TREE
//   Non utilizza gli alberi, meno memoria richiesta ma piu' lento
//   complessita': per ogni bit in input e' costante

// semplice test ad ottimizzazioni attivate per file di 4K
// notree | tree   | bitrate
// 60 sec | 45 sec | 10
// 31 sec | 23 sec | 16

// in:
//   Adiz dizionario da utilizzare, viene modificato esclusivamente per
//        la creazione degli alberi di partizione
coder_dictionary::coder_dictionary(dictionary& Adiz) : diz(Adiz) {
  first = true;
  nbit = 0;
  mbit = 0;
  tree = 0;
#if defined( CODER_USE_TREE )  
  // crea gli alberi
  diz.mimic_get().partition_tree_set();
#endif  
}

// inserisce nel buffer di output il simbolo
void coder_dictionary::output(element_index symbol) {
  // testo dewl simbolo
  const char* symbol_char = diz.alpha_get()[ symbol ];
  // per ogni carattere
  for(;*symbol_char;++symbol_char) {
    unsigned c = (unsigned)(unsigned char)*symbol_char;
    // per ogni bit
    for(unsigned i=0;i<CHAR_BIT;++i)
      buffer.put( (c & (1 << i)) != 0 );
  }
}

void coder_dictionary::put(bool bit) {
  PRECONDITION( nbit < CHAR_BIT * sizeof(unsigned) );
  // aggiunge il bit alla maschera
  if (bit)
    mbit |= 1 << nbit;
  ++nbit;
  
  bool done = false;
  // ripete finche' e' possibile elaborare l'informazione
  while (!done) {
    // se e' la prima emissione
    if (first) {
      vector<context_next*>::const_iterator seed = diz.mimic_get().seed_get( diz.seeds_get(), nbit, mbit );
      if (seed != diz.seeds_get().end()) {
        // imposta il contesto corrente
        current = **seed;
        // manda in output il contesto
        for(unsigned i=0;i<context::n_get();++i) {
          element_index index = current.index_forward_get( i );
          output( index );
        }
        // non e' piu' il primo
        first = false;
        // reinizializza
        nbit = 0;
        mbit = 0;
      } else {
        // servono piu' bit
        done = true;
      }
    } else {
#if !defined( CODER_USE_TREE )
      // cerca il contesto
      context_next_set::const_iterator i = diz.mimic_get().find( current );
      if (i == diz.mimic_get().end()) {
        ERROR("dictionary error, unknow context");
      }
      // per ogni elemento del gruppo cerca quelli con il codice appropriato
      vector<const element*> tab;
      // presuppone che nella maggior parte dei casi non si superi la
      // dimensione di size()/2 elementi
      tab.reserve( (*i).size() / 2 ); 
      // contatore presenze totali
      unsigned presence = 0;
      element_set::const_iterator k = (*i).begin();
      while (k != (*i).end()) {
        if ( ((*k).nbit_get() == nbit) && ((*k).mbit_get() == mbit)) {
          tab.push_back( (const element*)&*k );
          presence += (*k).presence_get();
        }
        ++k;
      }
      // se  stato trovato l'elemento
      if (tab.size() != 0) {
        // elemento scelto
        vector<const element*>::iterator j = tab.begin();
        // se gli elemento sono pi di 1 se ne sceglie uno a caso
        if (tab.size() > 1) {
          int r = rand() % presence;
          for(;r >= (int)(*j)->presence_get();++j) 
            r -= (*j)->presence_get();
        } 
        PRECONDITION( j != tab.end() );
        // output dell'elemento
        output( (*j)->index_get() );
        // nuovo contesto
        current.index_shift( (*j)->index_get() );
        // bit usati
        nbit = 0;
        mbit = 0;
      } else {
        // servono piu' bit
        done = true;
      }
#else
      // se e' un contesto nuovo
      if (!tree) {
	// cerca il contesto
        context_next_set::const_iterator i = diz.mimic_get().find( current );
        if (i == diz.mimic_get().end()) {
          ERROR("dictionary error, unknow context");
        }
        if ((*i).size() == 0) {
          ERROR( "dictionary error, no next" );
        }
	// albero del contesto
        tree = (*i).tree_get();
      }
      // legge bit in input fino a raggiungere la foglia dell'albero
      while (nbit>0 && !tree->leaf()) {
        tree = tree->put( (mbit & 1) != 0 );
	mbit >>= 1;
	--nbit;
      }
      // se ha raggiunto la foglia
      if (tree->leaf()) {
        // elemento scelto
        vector<const element*>::const_iterator j = tree->begin();
        // se gli elemento sono pi di 1 se ne sceglie uno a caso
        if (tree->size() > 1) {
          // iteratore dei contesti possibili
          vector<const element*>::const_iterator k = tree->begin();
	  // contatore di presenze totali
          unsigned presence = 0;
          while (k != tree->end()) {
            presence += (*k)->presence_get();
	    ++k;
	  }
          // estrae a caso
          int r = rand() % presence;
          for(;r >= (int)(*j)->presence_get();++j)
            r -= (*j)->presence_get();
        } 
        // output dell'elemento
        output( (*j)->index_get() );
        // nuovo contesto
        current.index_shift( (*j)->index_get() );
        // nuovo albero
        tree = 0;
      } else {
	// servono piu' bit in input
	done = true;
      }
#endif
    }
  }
}

bool coder_dictionary::internal_empty() const {
  if (first)
    return nbit == 0;
  else {
    PRECONDITION( tree );
    context_next_set::const_iterator i = diz.mimic_get().find( current );
    PRECONDITION( i != diz.mimic_get().end() );
    return tree == (*i).tree_get();
  }
}

void coder_dictionary::close() {
  // inserisce bit nulli fino all'emissione totale
  while (!internal_empty())  {
    put(false);
  }
  // chiude il buffer di output
  buffer.close();
}

bool coder_dictionary::get() {
  return buffer.get();
}

bool coder_dictionary::empty() const { 
  return buffer.empty();
}
  
state* coder_dictionary::tell() const {
  PRECONDITION( empty() );
  coder_dictionary_state* s = new coder_dictionary_state( buffer.tell() );
  s->first = first;
  s->current = current;
  s->nbit = nbit;
  s->mbit = mbit;
  s->tree = tree;
  return s;
}

void coder_dictionary::seek(const state* s) {
  buffer.seek( ((const coder_dictionary_state*)s)->buffer );
  first = ((const coder_dictionary_state*)s)->first;
  current = ((const coder_dictionary_state*)s)->current;
  nbit = ((const coder_dictionary_state*)s)->nbit;
  mbit = ((const coder_dictionary_state*)s)->mbit;
  tree = ((const coder_dictionary_state*)s)->tree;
}

