// $Log: mimic.h,v $
// Revision 1.6  1997/09/03 07:12:32  am
// Nuova gestione a stream di caratteri
//
// Revision 1.5  1997/09/02 21:34:21  am
// put e close non gestiscono piu' la condizione di errore
//
// Revision 1.4  1997/06/12 20:39:14  am
// aggiunto $Log: mimic.h,v $
// aggiunto Revision 1.6  1997/09/03 07:12:32  am
// aggiunto Nuova gestione a stream di caratteri
// aggiunto
// aggiunto Revision 1.5  1997/09/02 21:34:21  am
// aggiunto put e close non gestiscono piu' la condizione di errore
// aggiunto
//

#if !defined( __MIMIC_H )
#define __MIMIC_H

#pragma interface

#include <utility>
#include <algorithm>
#include <set>
#include <vector>
#include <map>

#include <debug.h>

#include <heap.h>
#include <iobit.h>
#include <dstream.h>

// --------------------------------------------------------------------------
// element_index

// indice per gli elementi
// note:
//   deve poter contenere il massimo numero di elementi
typedef unsigned element_index;

// ---------------------------------------------------------------------------
// element

class element {
  private:
    // codice elemento
    element_index index;
    // somma presenze
    unsigned presence;
    // codice di bit assegnato
    unsigned short mbit;
    // numero di bit del codice assegnato
    unsigned short nbit;
  
  public:
    element();
    element(const element&);
    element(element_index Aindex);
  
    element& operator=(const element&);
    bool operator==(const element&) const;
    bool operator<(const element&) const;

    void bit_set(unsigned Anbit, unsigned Ambit) ;
    unsigned mbit_get() const;
    unsigned nbit_get() const;

    void index_set(element_index index);
    element_index index_get() const;

    // presenze del elemento
    void presence_reset();
    void presence_inc() ;
    unsigned presence_get() const;

    friend ostream& operator<<(ostream&, const element&);
    friend istream& operator>>(istream&, element&);
  };

ostream& operator<<(ostream&, const element&);
istream& operator>>(istream&, element&);

inline element& element::operator=(const element& A) {
  index = A.index;
  presence = A.presence;
  nbit = A.nbit;
  mbit = A.mbit;
  return *this;
}

inline void element::index_set(element_index Aindex) {
  index = Aindex;
}
    
inline element_index element::index_get() const {
  return index;
}

inline void element::presence_reset() { 
  presence = 0; 
}

inline void element::presence_inc() { 
  ++presence; 
}

inline unsigned element::presence_get() const { 
  return presence; 
}

inline void element::bit_set(unsigned Anbit, unsigned Ambit) { 
  nbit = Anbit; mbit = Ambit; 
}

inline unsigned element::mbit_get() const { 
  return mbit; 
}

inline unsigned element::nbit_get() const { 
  return nbit; 
}

inline bool element::operator==(const element& A) const {
  return index == A.index;
}

inline bool element::operator<(const element& A) const {
  return index < A.index;
}

// ----------------------------------------------------------------------------

// contesto

// minima lunghezza del contesto
#define CONTEXT_MIN 1
// massima lunghezza del contesto
#define CONTEXT_MAX 8

class context {
    // lunghezza del contesto attiva, (non accedere direttamente)
    static unsigned n;
  protected:
    // contesto, elemento a indice 0  quello appena precedente
    element_index pred[CONTEXT_MAX];
  public:
    // imposta lunghezza
    static void n_set(unsigned An) {
      PRECONDITION( An >= CONTEXT_MIN && An <= CONTEXT_MAX );
      n = An;
    }

    // legge lunghezza
    static unsigned n_get() {
      return n;
    }
  
    context();
    context(const context& A);
  
    context& operator=(const context& A);
  
    void index_shift(element_index index);
    element_index index_back_get(int v = 0) const;
    element_index index_forward_get(int v = 0) const;

    bool operator<(const context& A) const;
    bool operator==(const context& A) const;
  };

inline element_index context::index_back_get(int v) const { 
  PRECONDITION( v>=0 && v<(int)n_get() ); 
  return pred[v]; 
}

inline element_index context::index_forward_get(int v) const { 
  return index_back_get( n_get()-1-v ); 
}

// --------------------------------------------------------------------------
// context_tree

class context_next;

class context_tree {
    context_tree(const context_tree&);
    context_tree& operator=(const context_tree&);
  
    // sotto alberi  
    context_tree* bit0;
    context_tree* bit1;
    // elementi
    vector<const element*> data;
  public:
    context_tree();
    context_tree(context_tree* Abit0, context_tree* Abit1);
    ~context_tree();
  
    void insert(const element* A);
    vector<const element*>::const_iterator begin() const { return data.begin(); }
    vector<const element*>::const_iterator end() const { return data.end(); }
    unsigned size() const { return data.size(); }
    
    bool leaf() const;
    const context_tree* put(bool bit) const;
  
#ifdef HEAP  
    static allocator_fixed allocator;

    void* operator new(size_t size) {
      return allocator.allocate( size );
    }
    void operator delete(void* p) {
      allocator.deallocate( p );
    }
#endif  
  
  }; 

inline bool context_tree::leaf() const { 
  return bit0 == 0; 
}

inline const context_tree* context_tree::put(bool bit) const {
  PRECONDITION( bit1 && bit0 );
  return bit ? bit1 : bit0;
}

// --------------------------------------------------------------------------
// context_next

// contesto con elementi successivi

//#define ELEMENT_SET_IS_SET
//   Occupazione notevole di memoria, dipendente dall'implementazione STL
//   Inserimento logaritmico
#undef ELEMENT_SET_IS_SET
//   Pi lento rispetto a set (presupponendo di avere abbastanza memoria) 
//   nella operazione di import a causa delle continue riallocazioni 
//   Inserimento lineare

#ifdef ELEMENT_SET_IS_SET
typedef less<element> element_compare;
typedef set<element,element_compare> element_set;
#else
typedef vector<element> element_set;
#endif

class context_next : public context {
  protected:
    // lista di elementi
    element_set data;

    // albero di riferimenti agli elementi successivi
    context_tree* tree;
  public:
    context_next();
    context_next(const context_next& A);
    context_next(const context& A);

    context_next& operator=(const context_next& A); 
  
    unsigned size() const;
  
    void presence_reset();
    unsigned presence_get() const;

    void partition_bit_set();
    void partition_tree_set();
    const context_tree* tree_get() const;
  
    // accesso agli elementi successivi al contesto
    element_set::iterator begin();
    element_set::iterator end();
    element_set::iterator find(const element& A);
    element_set::const_iterator begin() const;
    element_set::const_iterator end() const;
    element_set::const_iterator find(const element& A) const;
    pair<element_set::iterator,bool> insert(const element& A);
    void erase();
    void erase(element_set::iterator i);
  
    friend ostream& operator<<(ostream&, const context_next&);
    friend istream& operator>>(istream&, context_next&);
  };

ostream& operator<<(ostream&, const context_next&);
istream& operator>>(istream&, context_next&);

inline const context_tree* context_next::tree_get() const {
  PRECONDITION( tree );
  return tree;
}

inline element_set::iterator context_next::begin() { 
  return data.begin(); 
}

inline element_set::iterator context_next::end() { 
  return data.end(); 
}

inline element_set::iterator context_next::find(const element& A) { 
#ifdef ELEMENT_SET_IS_SET
  return data.find( A ); 
#else  
  element_set::iterator i = lower_bound( data.begin(), data.end(), A );
  if (i == data.end() || !(*i == A))
    return data.end();
  else
    return i;
#endif
}

inline element_set::const_iterator context_next::begin() const { 
  return data.begin(); 
}

inline element_set::const_iterator context_next::end() const { 
  return data.end(); 
}

inline element_set::const_iterator context_next::find(const element& A) const { 
#ifdef ELEMENT_SET_IS_SET
  return data.find( A ); 
#else  
  element_set::const_iterator i = lower_bound( data.begin(), data.end(), A );
  if (i == data.end() || !(*i == A))
    return data.end();
  else
    return i;
#endif  
}

inline pair<element_set::iterator,bool> context_next::insert( const element& A ) {
#ifdef ELEMENT_SET_IS_SET
  return data.insert( A );
#else  
  // alloca spazio per 1 elemento in pi
  data.reserve( data.size() + 1 );
  // posizione di inserimento
  element_set::iterator i = lower_bound( data.begin(), data.end(), A );
  if (i == data.end() || !(*i == A)) 
    return pair<element_set::iterator,bool>( data.insert( i,A ), true );
  else
    return pair<element_set::iterator,bool>( i , false );
#endif  
}

inline void context_next::erase() {
  data.erase( data.begin(), data.end() );
}

inline void context_next::erase(element_set::iterator i) {
  data.erase( i );
}

inline unsigned context_next::size() const {
  return data.size();
}

// -------------------------------------------------------------------------- 
// lstream

// stream di elementi

// note:
//   precondizione !eof per get

class iestream {
  public:
    virtual element_index get() = 0;
    virtual bool eof() = 0;
  };

class oestream {
  public:
    virtual void put(element_index c) = 0;
    virtual void flush() = 0;
  };

// ----------------------------------------------------------------------------

#define CONTEXT_SET_IS_SET
//  Utilizzo maggiore di memoria, inserimento logaritmico
//#undef CONTEXT_SET_IS_SET
//  ESTREMAMENTE lento nelle operazioni di import a causa dell'
//  inserimento lineare
//   

#ifdef CONTEXT_SET_IS_SET
typedef less<context_next> context_next_compare;
typedef set<context_next,context_next_compare> context_next_set;
#else
typedef vector<context_next> context_next_set;
#endif

class mimic {
    mimic& operator=(const mimic&);
    mimic(const mimic&);
  
    context seed_stego(const vector<context_next*>& seeds, ibstream& is) const;
    void seed_unstego(const context& seed, const vector<context_next*>& seeds, obstream& os) const;
  
  protected:
    context_next_set data;
  
    void shrink();
  
  public:
    mimic();	  
  
    void partition_bit_set();
    void partition_tree_set();
	  
    unsigned element_count() const;
    unsigned context_count() const;
    double bit_rate() const;

    vector<context_next*>::const_iterator seed_get(const vector<context_next*>& seeds, unsigned nbit, unsigned mbit) const; 
    void seed_put(const context& seed, const vector<context_next*>& seeds, unsigned& nbit, unsigned& mbit) const;
  
    void stego(const vector<context_next*>& seeds, oestream& os, ibstream& is);
    void unstego(const vector<context_next*>& seeds, obstream& Aos, iestream& Ais);
  
#ifdef ADAPTIVE  
    void stego_adaptive(const context& seed, oestream& os, ibstream& is);
    void unstego_adaptive(obstream& Aos, iestream& Ais);
#endif  
  
    void import(iestream& is);
  
    context_next_set::const_iterator begin() const { return data.begin(); }
    context_next_set::const_iterator end() const { return data.end(); }
    context_next_set::iterator begin() { return data.begin(); }
    context_next_set::iterator end() { return data.end(); }
    context_next_set::const_iterator find(const context_next& A) const; 
    pair<context_next_set::iterator,bool> insert(const context_next&);
  
    friend ostream& operator<<(ostream&, const mimic&);
    friend istream& operator>>(istream&, mimic&);
  };

inline context_next_set::const_iterator mimic::find(const context_next& A) const { 
#ifdef CONTEXT_SET_IS_SET
  return data.find( A ); 
#else  
  context_next_set::const_iterator i = lower_bound( data.begin(), data.end(), A );
  if (i == data.end() || !(*i == A))
    return data.end();
  else
    return i;
#endif  
}

inline pair<context_next_set::iterator,bool> mimic::insert(const context_next& A)  {
#ifdef CONTEXT_SET_IS_SET
  return data.insert( A );
#else  
  // posizione di inserimento
  context_next_set::iterator i = lower_bound( data.begin(), data.end(), A );
  if (i == data.end() || !(*i == A)) 
    return pair<context_next_set::iterator,bool>( data.insert( i,A ), true );
  else
    return pair<context_next_set::iterator,bool>( i , false );
#endif  
}

ostream& operator<<(ostream&, const mimic&);
istream& operator>>(istream&, mimic&);

// ---------------------------------------------------------------------------
// alpha

// alfabeto generico, associazione tra un simbolo (element_index) ed una 
// stringa (const char*)

struct string_compare {
    bool operator()(char* s1, char* s2) const {
      return strcmp( s1,s2 ) < 0;
    }
  };

typedef vector<char*> alpha_index;
typedef map<char*,element_index,string_compare> alpha_string;

class alpha {
    alpha(const alpha&);  
    alpha& operator=(const alpha&); 
  
    alpha_index mindex;
    alpha_string mstring;
  public:
    alpha();  
    ~alpha();
   
    element_index operator[](const char* A) const;
    element_index operator[](const char A) const;
    const char* operator[](element_index A) const;

    unsigned size() const;
  
    double bit_length() const;

    void erase();

    // stringhe
    bool has(const char* A) const;
    pair<bool,element_index> find(const char* A) const;
    element_index insert(const char* A);

    // stringhe di 1 solo carattere
    bool has(const char A) const;
    pair<bool,element_index> find(const char A) const;
    element_index insert(const char A);
  
    friend ostream& operator<<(ostream&, const alpha&);
    friend istream& operator>>(istream&, alpha&);
  };

inline unsigned alpha::size() const {
  return mindex.size();
}

inline bool alpha::has(const char* A) const {
  alpha_string::const_iterator i = mstring.find( (char*)A );
  return i != mstring.end();
}

inline bool alpha::has(const char A) const {
  char buf[2] = { A,0 };
  return has( buf );
}

inline pair<bool,element_index> alpha::find(const char* A) const {
  alpha_string::const_iterator i = mstring.find( (char*)A );
  if (i != mstring.end()) 
    return pair<bool,element_index>( true, (*i).second );
  else
    return pair<bool,element_index>( false, 0 );
}

inline pair<bool,element_index> alpha::find(const char A) const {
  char buf[2] = { A,0 };
  return find( buf );
}

inline element_index alpha::operator[](const char* A) const {
  alpha_string::const_iterator i = mstring.find( (char*)A );
  PRECONDITION( i != mstring.end() );
  return (*i).second;
}

inline element_index alpha::operator[](const char A) const {
  char buf[2] = { A,0 };
  return operator[](buf);
}

inline const char* alpha::operator[](element_index A) const {
  PRECONDITION( A < size() );
  return mindex[A];
}

inline element_index alpha::insert(const char A) {
  char buf[2] = { A,0 };
  return insert(buf);
}

ostream& operator<<(ostream&, const alpha&);
istream& operator>>(istream&, alpha&);

// ---------------------------------------------------------------------------
// dictionary

// interfaccia di un dizionario rappresentante una mimic function

class dictionary {
    dictionary(const dictionary&); 
    dictionary& operator=(const dictionary&);
  
    // in:
    //   diz,alp utilizzati per la scelta dei semi
    // out:
    //   inizializza seeds ai semi da utilizzare
    virtual void make_seeds() = 0;
  protected:
    // dizionario
    mimic diz;
    // alfabeto
    alpha alp;
    // semi
    vector<context_next*> seeds;
  public:
    dictionary(); 
    virtual ~dictionary(); 

    const alpha& alpha_get() const { return alp; }
    const mimic& mimic_get() const { return diz; }
    mimic& mimic_get() { return diz; }
    const vector<context_next*>& seeds_get() const { return seeds; }

    unsigned element_count() const; 
    unsigned context_count() const;
    unsigned alpha_count() const;
    double expansion_factor() const;
  
    virtual void stego(ostream& os, istream& is) = 0;
    virtual void unstego(ostream& os, istream& is) = 0;
    virtual void import(istream& is, unsigned n) = 0;
   
    friend ostream& operator<<(ostream&, const dictionary&);
    friend istream& operator>>(istream&, dictionary&);
  };

ostream& operator<<(ostream&, const dictionary&);
istream& operator>>(istream&, dictionary&);

// ---------------------------------------------------------------------------
// coder dictionary

// interfaccia per l'utilizzo di un dizionario rappresentante una mimic 
// function come un filtro codificatore

class coder_dictionary_state : public state {
    state* buffer;
    context current;
    bool first;
    unsigned nbit;
    unsigned mbit;
    const context_tree* tree;
    friend class coder_dictionary;
  public:
    coder_dictionary_state( state* Abuffer );
    virtual ~coder_dictionary_state();

#ifdef HEAP  
    static allocator_fixed allocator;

    void* operator new(size_t size) {
      return allocator.allocate( size );
    }
    void operator delete(void* p) {
      allocator.deallocate( p );
    }
#endif  
  };

class coder_dictionary : public siobit {
    // dizionario utilizzato
    dictionary& diz;
    // buffer di output
    iobit_buffer_static buffer;
    // contesto corrente
    context current;
    // codifica del seme ancora da effettuare
    bool first;
    // input non ancora elaborato
    unsigned nbit;
    unsigned mbit;
    // nodo dell'albero in elaborazione
    const context_tree* tree;
  
    virtual void output(element_index symbol);
    bool internal_empty() const;
  
  public:
    coder_dictionary(dictionary& Adiz);
  
    virtual void put(bool bit);
    virtual void close();
  
    virtual bool get();
    virtual bool empty() const;  
  
    virtual state* tell() const;
    virtual void seek(const state*);
  };

#endif
