// $Log: stego.cc,v $
// Revision 1.12  1997/09/06 07:45:57  am
// Aggiunta gestione debug
//
// Revision 1.11  1997/09/03 08:15:21  am
// Corretto --help
//
// Revision 1.10  1997/09/03 07:13:02  am
// Nuova gestione a stream di caratteri
// Inclusione di nospace in stego_hash
//
// Revision 1.9  1997/09/01 23:01:41  am
// Versione 0.6
//
// Revision 1.8  1997/08/26 15:59:02  am
// Inclusione condizionale di config.h
//
// Revision 1.7  1997/07/05 12:08:46  am
// Corretto bug sintattico in MSDOS
//
// Revision 1.6  1997/07/05 11:44:21  am
// Rivista completamente modalita' di gestione dizionario e output
// Differenziazione del comportamento per MSDOS
//
// Revision 1.5  1997/07/04 20:01:02  am
// forzata modalita' binaria anche per i file del
// testo steganografato
//
// Revision 1.4  1997/06/12 19:02:47  am
// interfaccia a riga di comando
//
// Revision 1.3  1997/06/06 06:53:56  am
// *** empty log message ***
//

#if !defined( __MSDOS__ )
#include <config.h>
#endif

#include <string.h>
#include <time.h>
#include <stdio.h>
#include <fstream.h>

#include <debug.h>
#include <word.h>
#include <letter.h>
#include <model.h>

// modalit binaria per i file
#ifdef __MSDOS__
#define IBIN ios::binary
#define OBIN ios::binary
#else
#define IBIN 0
#define OBIN 0
#endif

char* newstr(const char* s) {
  unsigned l = strlen(s);
  char* ss = new char[l+1];
  strcpy(ss,s);
  return ss;
}

// ----------------------------------------------------------------------------

// modalit di stego o unstego
enum { 
   stego,
   unstego,
   stego_model,
   unstego_model
 } mode = stego;             

unsigned model_rate = 0;

// tipo di steganografia
enum {
   unknow,
   word,
   letter
  } type = unknow;
 
//  stata fatta una operazione tale da inibire l'utizzo degli standard input/output
bool operate = false;          

// ---------------------------------------------------------------------------
// help

const char help_msg[] =
"Options:\n"
"-s, --stego            stego mode\n"
"-x, --unstego          unstego mode\n"
"-S, --stego-hash N     stego with hash function at rate N mode\n"
"-X, --unstego-hash N   unstego with hash function at rate N mode\n"
"-f, --file FILE        output file is FILE\n"
"-d, --diz FILE         read dictionary from FILE\n"
"-iw, --import-word FILE N    (in word mode)\n" 
"-il, --import-letter FILE N  (in letter mode)\n"
"                       import dictionary from FILE, with dependence N\n"
"-w, --write FILE       write dictionary to FILE\n"
"-r, --rate             show bitrate for active dictionary\n"
"--version              version info\n"
"--help                 this help\n"
"\nDefault is: -s -d ./stego.diz\n";

void help() {
  cout << help_msg;
}

// ---------------------------------------------------------------------------
// dizionario

// file dizionario attivo, allocato dinamicamente
static char* diz_file;        
// dizionario, allocato dinamicamente
static dictionary* diz_data;           

// massima lunghezza della stringa indicante il tipo di dizionario
#define DIZ_TYPE_MAX 128

// ritorna il dizionario attuale
// return:
//   reference valida fino alla prossima chiamata diz_
dictionary& diz_get() {
  // se  vuoto 
  if (!diz_data) {
    const char* file = diz_file ? diz_file : "./stego.diz";
    ifstream is( file, ios::in );
    if (!is) {
      cerr << "error: load dictionary " << file << endl;
      exit(EXIT_FAILURE);
    } 
    char type_str[DIZ_TYPE_MAX];
    is.getline( type_str, sizeof(type_str) );
    if (strcmp( type_str, "word")==0) {
      type = word;
      diz_data = new dictionary_word;
    } else if (strcmp( type_str, "letter") == 0) {
      type = letter;
      diz_data = new dictionary_letter;
    } else {
      cerr << "error: unknow dictionary type" << endl;
      exit(EXIT_FAILURE);
    } 
    is >> *diz_data;
    is.close();
  }
  return *diz_data;
}

// importa il dizionario da un file
void diz_import(const char* file, unsigned n) {
  if (diz_file) delete [] diz_file;
  diz_file = 0;
  if (diz_data) delete diz_data;
  
  switch (type) {
    case word : 
      diz_data = new dictionary_word; 
      break;
    case letter : 
      diz_data = new dictionary_letter; 
      break;
    default:
      PRECONDITION( 0 );
  }
  
  if (n < CONTEXT_MIN || n > CONTEXT_MAX) {
    cerr << "error: dipendence out of range [" << CONTEXT_MIN << "," << CONTEXT_MAX << "]" << endl;
    exit(EXIT_FAILURE);
  }
  ifstream is( file, ios::in );
  if (!is) {
    cerr << "error: opening file " << file << endl;
    exit(EXIT_FAILURE);
  }
  // importa
  diz_data->import( is, n );
  
  is.close();
}

// imposta il nome del dizionario
void diz_set(const char* file) {
  //  lo stesso
  if (diz_file && strcmp(diz_file,file)==0)
    return;
  type = unknow;
  // dealloca
  if (diz_data) delete diz_data;
  diz_data = 0;
  // nuovo
  if (diz_file) delete [] diz_file;
  diz_file = newstr( file );
}

void diz_save(const char* file) {
  // forza lettura del dizionario
  diz_get();
  
  if (!diz_data) {
    cerr << "error: no dictionary in memory" << endl;
    exit(EXIT_FAILURE);
  }
  ofstream os( file, ios::out | ios::trunc );
  if (!os) {
    cerr << "error: opening file " << file << endl;
    exit(EXIT_FAILURE);
  }
  switch (type) {
    case word : 
      os << "word" << endl; 
      break;
    case letter : 
      os << "letter" << endl; 
      break;
    default:
     PRECONDITION( 0 );   
  }
  os << *diz_data;
  os.close();
}

void diz_init() {
  diz_file = 0;
  diz_data = 0;
}

void diz_done() {
  if (diz_file) delete [] diz_file;
  if (diz_data) delete diz_data;
}

// --------------------------------------------------------------------------
// gestione nomi file

// flag per la rimozione del file di input
static bool output_inremove;
// file di input
static char* output_infile;
// file di output
static char* output_outfile;

// estensione di default per i file di output steganografati
#define OUTPUT_EXT ".stego"
#define OUTPUT_EXT_LEN (sizeof(OUTPUT_EXT)-1)

// imposta il prossimo file di output
void output_set(const char* file) {
  PRECONDITION( file );
  if (output_outfile) {
    cerr << "error: output file " << output_outfile << " not used" << endl;
    exit( EXIT_FAILURE );
  }
  output_outfile = newstr( file  );
}

// richiesta di un nome file di output
// in:
//   file file di input
//   stego flag indicante se e' una operazione  di stego o unstego
#if !defined( __MSDOS__ )
void output_request(const char* file, bool stego) {
  if (!output_outfile) {
    if (stego) {
      output_outfile = new char[ strlen(file) + OUTPUT_EXT_LEN + 1 ];
      strcpy( output_outfile, file);
      strcat( output_outfile, OUTPUT_EXT );
    } else {
      unsigned l = strlen(file);
      if (l>=OUTPUT_EXT_LEN && strcmp(file+l-OUTPUT_EXT_LEN,OUTPUT_EXT)==0) {
        output_outfile = new char[l-OUTPUT_EXT_LEN+1];
        memcpy(output_outfile,file,l-OUTPUT_EXT_LEN);
        output_outfile[l-OUTPUT_EXT_LEN] = 0;
      } else {
        cerr << "error: no output file for " << file << endl;
        exit(EXIT_FAILURE);
      }
    }
    output_inremove = true;
    PRECONDITION( !output_infile );
    output_infile = newstr( file );
  } else
    output_inremove = false;
#else 
void output_request(const char*, bool) {
  if (!output_outfile) {
    cerr << "error: no output file specified" << endl;
    exit( EXIT_FAILURE );
  }
  PRECONDITION( !output_infile );
  output_inremove = false;
#endif  
  PRECONDITION( output_outfile && (output_inremove == (output_infile!=0)) );
}

// file di output assegnato
// return:
//   file di output assegnato da output_request
const char* output_file_get() {
  return output_outfile;
}

// terminazione dell'uso del file di output dato assegnato da output request
void output_used() {
  if (output_outfile) {
    delete [] output_outfile;
    output_outfile = 0;
  }
  if (output_inremove) {
    if (remove( output_infile )) {
      cerr << "error: remove " << output_infile << endl;
      exit(EXIT_FAILURE);
    }
    delete [] output_infile;
    output_infile = 0;
  }
  PRECONDITION( !output_infile && !output_outfile );
}

void output_init() {
  output_inremove = false;
  output_infile = 0;
  output_outfile = 0;
}

void output_done() {
  PRECONDITION( !output_infile );
  if (output_outfile) {
    cerr << "error: output file " << output_outfile << " not used" << endl;
    delete [] output_outfile;
    exit( EXIT_FAILURE );
  }
  PRECONDITION( !output_infile && !output_outfile );
}

// --------------------------------------------------------------------------
// operazioni

void file_stego(const char* file, dictionary& diz) { 
  output_request( file, true );
  ifstream is(file, ios::in | IBIN );
  if (!is) {
    cerr << "error: opening file " << file << endl;
    exit(EXIT_FAILURE);
  }
  ofstream os( output_file_get(), ios::out | ios::trunc | OBIN );
  if (!os) {
    cerr << "error: opening file " << output_file_get() << endl;
    exit(EXIT_FAILURE);
  }
  cout << file << ": " << output_file_get() << endl;
  
  diz.stego(os,is);
  
  os.close();
  is.close();
  output_used();
}

void file_unstego(const char* file, dictionary& diz) {
  output_request( file, false );
  ifstream is( file, ios::in | IBIN );
  if (!is) {
    cerr << "error: opening file " << file << endl;
    exit(EXIT_FAILURE);
  }
  ofstream os(output_file_get(), ios::out | ios::trunc | OBIN );
  if (!os) {
    cerr << "error: opening file " << output_file_get() << endl;
    exit(EXIT_FAILURE);
  }
  cout << file << ": "  << output_file_get() << endl;
  
  // decodifica
  diz.unstego(os,is);
  
  os.close();
  is.close();
  output_used();
}

void stream_stego_hash(ostream& os, istream& is, dictionary& diz) {
  icfstream i1(is);
  ibstream isb(i1);
  
  ocfstream o1(os);
  ocwrapstream o2(o1);
  obstream osb(o2);
  
#if 1
  // codificatore con dizionario
  coder_dictionary c(diz);
#else  
  // connessione con il codificatore a dizionario ed il word wrapper 
  ioconnect c( new coder_dictionary(diz), new iochar2bit( new word_wrap(WRAP) ));
#endif  
  
  // decodificatore
  decoder_standard d(model_rate);
  // codificatore
  model m(c,d,32);

  // inserisce informazioni nel codificatore
  while (!isb.eof())
    m.put( isb.get() );
  m.close();
  
  // estrae informazioni dal codificatore
  while (!m.empty()) 
    osb.put( m.get() );
  osb.flush();
  
}

void file_stego_model(const char* file, dictionary& diz) { 
  output_request( file, true );
  ifstream is(file, ios::in | IBIN );
  if (!is) {
    cerr << "error: opening file " << file << endl;
    exit(EXIT_FAILURE);
  }
  ofstream os( output_file_get(), ios::out | ios::trunc | OBIN );
  if (!os) {
    cerr << "error: opening file " << output_file_get() << endl;
    exit(EXIT_FAILURE);
  }
  cout << file << ": " << output_file_get() << endl;

  // codifica
  stream_stego_hash( os, is, diz );
  
  os.close();
  is.close();
  output_used();
}

void stream_unstego_hash(ostream& os, istream& is) {
  icfstream i1(is);
  icunwrapstream i2(i1);
  ibstream isb(i2);
  
  ocfstream o1(os);
  obstream osb(o1);
  
  decoder_standard d(model_rate);

  // inserisce informazioni nel decoder
  while (!isb.eof()) {
    d.put( isb.get() );
    while (!d.empty())
     osb.put( d.get() );
  }
  d.close();
  
  // estrae informazioni dal decoder
  while (!d.empty())
    osb.put( d.get() );
  osb.flush();
}

void file_unstego_model(const char* file) {
  output_request( file, false );
  ifstream is( file, ios::in | IBIN);
  if (!is) {
    cerr << "error: opening file " << file << endl;
    exit(EXIT_FAILURE);
  }
  ofstream os(output_file_get(), ios::out | ios::trunc | OBIN );
  if (!os) {
    cerr << "error: opening file " << output_file_get() << endl;
    exit(EXIT_FAILURE);
  }
  cout << file << ": "  << output_file_get() << endl;

  // decodfica
  stream_unstego_hash( os,is );

  os.close();
  is.close();
  output_used();
}

// ---------------------------------------------------------------------------
// opzioni

bool opt(const char* v, const char* o1, const char* o2) {
  return strcmp(v,o1) == 0 || strcmp(v,o2)==0;
}

bool opt(const char* v, const char* o1) {
  return strcmp(v,o1) == 0;
}

void process(int argc, char* argv[]) {
  for(int i=0;i<argc;++i) {
    if (opt(argv[i],"--help")) {
      operate = true;
      help();
    } else if (opt(argv[i],"--version")) {
      operate = true;
      cout << "stego v0.6 1997 by /_\\|\\/|" << endl;   
      cout << "BETA VERSION" << endl;
    } else if (opt(argv[i],"--stego","-s")) {
      mode = stego;
    } else if (opt(argv[i],"--unstego","-x")) {
      mode = unstego;
    } else if (opt(argv[i],"--hash","-S")) {
      mode = stego_model;
      unsigned n = atoi( argv[i+1] );
      if (n<2 || n>1024) {
        cerr << "syntax: " << argv[i] << " N" << endl;
        exit(EXIT_FAILURE);
      }
      model_rate = n;
      ++i;
    } else if (opt(argv[i],"--unhash","-X")) {
      mode = unstego_model;
      unsigned n = atoi( argv[i+1] );
      if (n<2 || n>1024) {
        cerr << "syntax: " << argv[i] << " N" << endl;
        exit(EXIT_FAILURE);
      }
      model_rate = n;
      ++i;
    } else if (opt(argv[i],"--dictionary","-d")) {
      if (i+1>=argc) {
        cerr << "syntax: " << argv[i] << " FILE" << endl;
        exit(EXIT_FAILURE);
      }
      diz_set( argv[i+1] );
      ++i;
    } else if (opt(argv[i],"--rate","-r")) {
      operate = true;
      cout << "dictionary: expansion factor " << diz_get().expansion_factor() << endl;
    } else if (opt(argv[i],"--import-word","-iw")) {
      if (i+2>=argc) {
        cerr << "syntax: " << argv[i] << " FILE N" << endl;
        exit(EXIT_FAILURE);
      }
      unsigned n = atoi( argv[i+2] );
      type = word;
      diz_import( argv[i+1], n );
      if (!diz_get().context_count()) {
        cerr << "error: dictionary is empty" << endl;
        exit(EXIT_FAILURE);
      }
      i += 2;
    } else if (opt(argv[i],"--import-letter","-il")) {
      if (i+2>=argc) {
        cerr << "syntax: " << argv[i] << " FILE N" << endl;
        exit(EXIT_FAILURE);
      }
      unsigned n = atoi( argv[i+2] );
      type = letter;
      diz_import( argv[i+1], n );
      if (!diz_get().context_count()) {
        cerr << "error: dictionary is empty" << endl;
        exit(EXIT_FAILURE);
      }
      i += 2;
    } else if (opt(argv[i],"--write","-w")) {
      operate = true;
      if (i+1>=argc) {
        cerr << "syntax: " << argv[i] << " FILE" << endl;
        exit(EXIT_FAILURE);
      }
      cout << "dictionary: " << argv[i+1] << endl;
      diz_save( argv[i+1] );
      ++i;
    } else if (opt(argv[i],"--file","-f")) {
      operate = true;
      if (i+1>=argc) {
        cerr << "syntax: " << argv[i] << " output_file" << endl;
        exit(EXIT_FAILURE);
      }
      output_set( argv[i+1] );
      ++i;
    } else if (argv[i][0]=='-') {
       cerr << "syntax: wrong parameter " << argv[i] << endl;
       exit(EXIT_FAILURE);
    } else {
     operate = true;
     switch (mode) {
       case stego : file_stego(argv[i], diz_get()); break;
       case unstego : file_unstego(argv[i], diz_get() ); break;
       case stego_model : file_stego_model(argv[i], diz_get()); break;
       case unstego_model : file_unstego_model(argv[i]); break;
     }
    }
  }
}

void process_env() {
  char* s = getenv("STEGO");
  if (s) {
    // duplica la stringa
    char* line = newstr( s );
    // crea una struttura argc/v
    unsigned argmax = 16;
    char** argv = new char*[argmax];
    unsigned argc = 0;
    char* param = strtok(line," ");
    while (param) {
      if (argc+1>=argmax) {
	unsigned nargmax = argmax * 2;
	char** nargv = new char*[nargmax];
	for(unsigned i=0;i<argc;++i)
	  nargv[i] = argv[i];
	delete [] argv;
	argv = nargv;
	argmax = nargmax;
      }
      argv[argc++] = param;
      param = strtok(NULL," ");
    }
    argv[argc] = 0;
    // processa la struttura argv argc
    process(argc,argv);
    delete [] argv;
    delete [] line;
  }
}

int main(int argc, char* argv[]) {
  
  DEBUG_INIT();
  
  // inizializza
  srand( time(0) );
     
  diz_init();
  output_init();
  
  // processa opzioni in environment
  process_env();
  // provessa opzioni sulla riga di comando
  process(argc-1,argv+1);

#if defined( __MSDOS__ )  
  if (!operate) {
    cerr << "error: no operation (try: stego --help)" << endl;
    exit( EXIT_FAILURE );
  }
#else  
  // se non  stata fatta nessuna operazione procede alla gestione
  // degli standard input/ouput
  if (!operate) {
    switch (mode) {
      case stego: 
        diz_get().stego(cout,cin); 
        break;
      case unstego: 
        diz_get().unstego(cout,cin); 
        break;
      case stego_model: 
        stream_stego_hash(cout,cin, diz_get()); 
        break;
      case unstego_model : 
        stream_unstego_hash(cout,cin); 
        break;
    }
  }
#endif

  // deinizializza
  output_done();
  diz_done();

  DEBUG_DONE();
  
  return EXIT_SUCCESS;
}
